<?php

class Api_Controller extends MVC_Controller
{
	public function index()
	{
        $this->header->allow();

        if(!$this->session->has("logged"))
            response(401);

        $this->cache->container("system.settings");

        if($this->cache->empty()):
            $this->cache->setArray($this->system->getSettings());
        endif;

        set_system($this->cache->getAll());

        set_logged($this->session->get("logged"));

        $vars = [
            "site_url" => site_url(false, true),
            "data" => [
                "number" => $this->phone->getExampleNumber(logged_country, Brick\PhoneNumber\PhoneNumberType::MOBILE),
                 "type" => "default"
            ]
        ];
        
        $this->smarty->display("_apidoc/layout.tpl", $vars);
	}

    public function send()
    {
        $this->header->allow();

        $this->cache->container("system.settings");

        if($this->cache->empty()):
            $this->cache->setArray($this->system->getSettings());
        endif;

        set_system($this->cache->getAll());

        $request = $this->sanitize->array($_REQUEST);
        $service = $this->sanitize->string($this->url->segment(4));

        if(!isset($request["secret"]))
            response(400, "Invalid Parameters!");

        if($this->api->checkApikey($request["secret"]) < 1)
            response(401, "Invalid API secret supplied!");

        $api = $this->api->getApikey($request["secret"]);

        switch($service):
            case "otp":
                /**
                 * @api {post} /send/otp Send OTP
                 * @apiName Send OTP
                 * @apiDescription Send a one-time-password to specified mobile number. Requires "<strong>otp</strong>" API permission.
                 * @apiGroup OTP
                 * @apiVersion 1.0.0
                 *
                 * @apiParam {String} secret The API secret you copied from (Tools -> API Keys) page
                 * @apiParam {String="sms","whatsapp"} type Type of message, it can be SMS or WhatsApp.
                 * @apiParam {String} message OTP message to send, you can use <strong>{{otp}}</strong> shortcode to include the otp anywhere in the message.
                 * @apiParam {String} phone Recipient mobile number, it will accept E.164 formatted numbers<br>
                 * <strong>Example for Philippines</strong><br>
                 * E.164: +639184661533
                 * @apiParam {Number} expire OTP expiration time in seconds. This is optional, default value is 300 seconds or 5 minutes.
                 * @apiParam {String="devices","credits"} mode This is only for <strong>sms</strong> type. This is the mode of sending the message, it can be "devices" which will allow you to use your linked android devices or "credits" which will allow you to use gateways and partner devices. "credits" requires you to have enough credit balance to send messages.
                 * @apiParam {String} [device] This is only for <strong>sms</strong> type. Linked device unique ID, this is required if you will send with "devices" mode. You can get linked device unique ID from <strong>/get/devices</strong> (Your devices).
                 * @apiParam {String|Number} [gateway] This is only for <strong>sms</strong> type. Partner device unique ID or gateway ID, this is required if you will send with "credits" mode. You can get a partner device unique ID and gateway ID from <strong>/get/rates</strong> 
                 * @apiParam {Number=1,2} [sim] This is only for <strong>sms</strong> type. Sim slot number you want to use. For "devices" mode only.
                 * @apiParam {Number} account This is only for <strong>whatsapp</strong> type. WhatsApp account ID you want to use for sending, you can get account ID's from <strong>/get/wa.accounts</strong> (Your WhatsApp accounts).
                 *
                 * @apiExample {php} PHP Example
                  <?php

                    $message = [
                        "secret" => "API_SECRET", // your API secret from (Tools -> API Keys) page
                        "mode" => "sms",
                        "mode" => "devices",
                        "device" => "00000000-0000-0000-d57d-f30cb6a89289",
                        "sim" => 1,
                        "phone" => "+639123456789",
                        "message" => "Your OTP is {{otp}}"
                    ];

                    $cURL = curl_init("http://127.0.0.1/zender/api/send/sms");
                    curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    curl_setopt($cURL, CURLOPT_POSTFIELDS, $message);
                    $response = curl_exec($cURL);
                    curl_close($cURL);

                    $result = json_decode($response, true);

                    // do something with response
                    print_r($result);
                 * 
                 * @apiExample {python} Python Example
                    import requests
    
                    # your API secret from (Tools -> API Keys) page
                    apiSecret = "API_SECRET"

                    message = {
                        "secret": apiSecret,
                        "type": "sms",
                        "mode": "devices",
                        "device": "00000000-0000-0000-d57d-f30cb6a89289",
                        "sim": 1,
                        "phone": "+639123456789",
                        "message": "Your OTP is {{otp}}"
                    }

                    r = requests.post(url = "http://127.0.0.1/zender/api/send/sms", params = message)
                      
                    # do something with response object
                    result = r.json()
                 *
                 * @apiSuccess (Success Response Format) {Number} status List of Codes
                 * <br> 200 = Success
                 * @apiSuccess (Success Response Format) {String} message Response message
                 * @apiSuccess (Success Response Format) {Array} data Array of data
                 *
                 * @apiSuccessExample {json} Success Response
                 {
                   "status": 200,
                   "message": "OTP has been sent!",
                   "data": {
                        phone: "+639123456789",
                        message: "Your OTP is 345678",
                        otp: 345678
                   }
                 }
                 * 
                 * @apiError (Error Response Format) {Number} status List of Codes<br>
                 * 400 = Invalid parameters<br>
                 * 401 = Invalid API secret<br>
                 * 403 = Access denied<br>
                 * 404 = Device doesn't exist<br>
                 * 500 = Something went wrong
                 * @apiError (Error Response Format) {String} message Response message
                 * @apiError (Error Response Format) {Array} data Array of data
                 * 
                 * @apiErrorExample {json} Error Response
                 {
                   "status": 400,
                   "message": "Invalid Parameters!",
                   "data": false
                 }
                 *
                 */

                if(!in_array("otp", explode(",", $api["permissions"])))
                    response(403, "This API key doesn't have permission to use this endpoint!");

                if(!isset($request["message"]))
                    response(400, "Invalid Parameters!");

                if(isset($request["expire"])):
                    if(!$this->sanitize->isInt($request["expire"])):
                        response(400, "Invalid Parameters!");
                    endif;
                endif;

                $this->cache->container("api.otp.{$request["secret"]}", true);

                $unique = false;

                while(!$unique):
                    $otp = rand(100000, 999999);

                    if(!$this->cache->has($otp)):
                        $this->cache->set($otp, true, isset($request["expire"]) ? ($request["expire"] < 1 ? 1 : $request["expire"]) : 300);

                        $unique = true;
                    endif;
                endwhile;

                $request["message"] = $this->lex->parse($request["message"], [
                    "otp" => $otp
                ]);

                if($request["type"] == "sms"):
                    if(!isset($request["mode"], $request["phone"]))
                        response(400, "Invalid Parameters!");

                    if(!$this->file->exists("system/storage/temporary/firebase.json")):
                        response(500, "System configuration error!");
                    endif;

                    try {
                        $number = $this->phone->parse($request["phone"], $api["country"]);

                        $number->format(Brick\PhoneNumber\PhoneNumberFormat::INTERNATIONAL);

                        if(!$number->isValidNumber())
                            response(400, "Invalid phone number!");

                        if(!$number->getNumberType(Brick\PhoneNumber\PhoneNumberType::MOBILE))
                            response(400, "Invalid phone number!");

                        $request["phone"] = $number->format(Brick\PhoneNumber\PhoneNumberFormat::E164);
                        $country = $number->getRegionCode();
                    } catch(Brick\PhoneNumber\PhoneNumberParseException $e) {
                        response(400, "Invalid phone number!");
                    }

                    if(!$this->sanitize->length($request["message"], system_message_min))
                        response(400, "OTP message is too short!");

                    if(system_message_max > 0):
                        if($this->sanitize->length($request["message"], system_message_max, 2))
                            response(400, "OTP message is too long!");
                    endif;

                    if($request["mode"] == "devices"):
                        if(!isset(
                            $request["device"],
                            $request["sim"]
                        ))
                            response(400, "Invalid Parameters!");

                        $subscription = set_subscription(
                            $this->system->checkSubscription($api["uid"]), 
                            $this->system->getSubscription(false, $api["uid"]), 
                            $this->system->getSubscription(false, false, true)
                        );

                        if(empty($subscription))
                            response(403, "Account is not subscribed to any premium package!");

                        if(!$this->sanitize->isInt($request["sim"]))
                            response(400, "Invalid Parameters!");

                        if(limitation($subscription["send_limit"], $this->system->countQuota($api["uid"], "sent")))
                            response(403, "Maximum allowed number of sent messages has been reached!");

                        if($this->system->checkDevice($api["uid"], $request["device"], "did") < 1)
                            response(404, "Device doesn't exist!");

                        $device = $this->system->getDevice($api["uid"], $request["device"], "did");
                    else:
                        if($request["mode"] != "credits")
                            response(400, "Invalid Parameters!");

                        if(!isset($request["gateway"]))
                            response(400, "Invalid Parameters!");

                        if($this->sanitize->isInt($request["gateway"])):
                            $gateways = $this->system->getGateways();

                            if(!array_key_exists($request["gateway"], $gateways)):
                                response(404, "Specified gateway doesn't exist!");
                            endif;

                            if(!$this->file->exists("system/gateways/" . md5($request["gateway"]) . ".php"))
                                response(404, "Specified gateway doesn't exist!");

                            require "system/gateways/" . md5($request["gateway"]) . ".php";
                        else:
                            $device = $this->system->getDevice(false, $request["gateway"], "global");

                            if($device):
                                if($device["global_device"] > 1):
                                    response(403, "Device is not available!");
                                endif;
                            else:
                                response(404, "Device doesn't exist!");
                            endif;
                        endif;
                    endif;

                    if($request["mode"] == "devices"):
                        $this->system->create("sent", [
                            "uid" => $api["uid"],
                            "did" => $request["device"],
                            "gateway" => 0,
                            "sim" => $request["sim"] < 2 ? 1 : 2,
                            "mode" => 1,
                            "phone" => $request["phone"],
                            "message" => bjoernffm\Spintax\Parser::parse(footermark($subscription["footermark"], $request["message"], system_message_mark))->generate(),
                            "status" => 1,
                            "status_code" => false,
                            "priority" => 1,
                            "api" => 1,
                            "create_date" => date("Y-m-d H:i:s", time())
                        ]);
                    else:
                        $credits = $this->system->getCredits($api["uid"]);

                        if($this->sanitize->isInt($request["gateway"])):
                            $pricing = json_decode($gateways[$request["gateway"]]["pricing"], true);

                            if(array_key_exists(strtolower($country), $pricing["countries"])):
                                $price = $pricing["countries"][strtolower($country)];
                            else:
                                $price = $pricing["default"];
                            endif;

                            if($credits < $price)
                                response(403, "Not enough credits to send this message!");

                            $gateway = $gateways[$request["gateway"]];

                            $message = bjoernffm\Spintax\Parser::parse($request["message"])->generate();

                            $send = gatewaySend($request["phone"], $message, $this);

                            if($send):
                                $create = $this->system->create("sent", [
                                    "uid" => $api["uid"],
                                    "did" => false,
                                    "gateway" => $request["gateway"],
                                    "api" => 1,
                                    "sim" => 0,
                                    "mode" => 2,
                                    "priority" => 0,
                                    "phone" => $request["phone"],
                                    "message" => $message,
                                    "status" => $gateway["callback"] < 2 ? 2 : 3,
                                    "status_code" => false
                                ]);

                                if($gateway["callback"] < 2):
                                    $this->cache->container("system.gateways");

                                    $this->cache->set("{$gateway["callback_id"]}.{$send}", $create);
                                endif;
                            else:
                                response(500, "Gateway was unable to send your message!");
                            endif;
                        else:
                            $currency = country($device["country"])->getCurrency()["iso_4217_code"];

                            $this->cache->container("system.payments", true);

                            if(!$this->cache->has("exchange")):
                                try {
                                    $exchange = json_decode($this->guzzle->get(titansys_api . "/currency?code=" . system_purchase_code, [
                                        "allow_redirects" => true,
                                        "http_errors" => false
                                    ])->getBody()->getContents(), true);

                                    if($exchange["status"] == 200):
                                        $this->cache->set("exchange", $exchange, 43200);
                                    else:
                                        response(500, "System configuration error!");
                                    endif;
                                } catch(Exception $e){
                                    response(500, "System configuration error!");
                                }
                            endif;

                            $rates = $this->cache->get("exchange");

                            $base_rate = $rates["data"]["USD"] / $rates["data"][strtoupper($currency)];
                            $usd_price = ($base_rate * $device["rate"]) * $rates["data"]["USD"];
                            $final_price = (float) abs($usd_price * $rates["data"][strtoupper(system_currency)]);

                            if($credits < $final_price)
                                response(403, "Not enough credits to send this message!");

                            $slots = explode(",", $device["global_slots"]);

                            $this->system->create("sent", [
                                "uid" => $api["uid"],
                                "did" => $request["gateway"],
                                "gateway" => 0,
                                "sim" => count($slots) > 1 ? rand(1, 2) : ($slots[0] < 2 ? 1 : 2),
                                "mode" => 2,
                                "phone" => $request["phone"],
                                "message" => bjoernffm\Spintax\Parser::parse($request["message"])->generate(),
                                "status" => 1,
                                "status_code" => false,
                                "priority" => $device["global_priority"],
                                "api" => 1,
                                "create_date" => date("Y-m-d H:i:s", time())
                            ]);
                        endif;
                    endif;

                    $this->echo->_cache = $this->cache;
                    $this->echo->_guzzle = $this->guzzle;

                    try {
                        $echoToken = $this->echo->token();

                        if($echoToken):
                            $this->echo->notify($api["hash"], [
                                "type" => "table"
                            ]);
                        endif;
                    } catch(Exception $e){
                        // Ignore
                    }

                    if($request["mode"] == "devices"):
                        if(round((int) $device["version"]) > 7):
                            $this->fcm->send(md5($api["uid"] . $request["device"]), [
                                "type" => "sms",
                                "global" => 0,
                                "currency" => "None",
                                "rate" => (float) 0
                            ]);
                        else:
                            try {
                                $echoToken = $this->echo->token();

                                if($echoToken):
                                    $this->echo->notify("legacy_" . md5($api["uid"] . $request["device"]), [
                                        "type" => "sms",
                                        "global" => 0,
                                        "currency" => "None",
                                        "rate" => (float) 0
                                    ]);
                                endif;
                            } catch(Exception $e){
                                // Ignore;
                            }
                        endif;
                    else:
                        if(!$this->sanitize->isInt($request["gateway"])):
                            if(round((int) $device["version"]) > 7):
                                $this->fcm->send(md5($device["uid"] . $request["gateway"]), [
                                    "type" => "sms",
                                    "global" => 1,
                                    "currency" => $currency,
                                    "rate" => (float) $device["rate"]
                                ]);
                            else:
                                try {
                                    $echoToken = $this->echo->token();

                                    if($echoToken):
                                        $this->echo->notify("legacy_" . md5($device["uid"] . $request["gateway"]), [
                                            "type" => "sms",
                                            "global" => 1,
                                            "currency" => $currency,
                                            "rate" => (float) $device["rate"]
                                        ]);
                                    endif;
                                } catch(Exception $e){
                                    // Ignore
                                }
                            endif;
                        endif;
                    endif;
                else:
                    if($request["type"] != "whatsapp")
                        response(400, "Invalid Parameters!");

                    if(!isset($request["phone"], $request["account"]))
                        response(400, "Invalid Parameters!");

                    $subscription = set_subscription(
                        $this->system->checkSubscription($api["uid"]), 
                        $this->system->getSubscription(false, $api["uid"]), 
                        $this->system->getSubscription(false, false, true)
                    );

                    if(empty($subscription))
                        response(403, "Account is not subscribed to any premium package!");

                    if($this->system->checkWaAccount($api["uid"], $request["account"], "id") < 1)
                        response(404, "WhatsApp account doesn't exist!");

                    try {
                        $number = $this->phone->parse($request["phone"], $api["country"]);

                        $number->format(Brick\PhoneNumber\PhoneNumberFormat::INTERNATIONAL);

                        if(!$number->isValidNumber())
                            response(400, "Invalid phone number!");

                        if(!$number->getNumberType(Brick\PhoneNumber\PhoneNumberType::MOBILE))
                            response(400, "Invalid phone number!");

                        $request["phone"] = $number->format(Brick\PhoneNumber\PhoneNumberFormat::E164);
                    } catch(Brick\PhoneNumber\PhoneNumberParseException $e) {
                        response(400, "Invalid phone number!");
                    }

                    if(limitation($subscription["wa_send_limit"], $this->system->countQuota($api["uid"], "wa_sent")))
                        response(403, "You have reached the maximum number of allowed chats!");

                    $this->echo->_cache = $this->cache;
                    $this->echo->_guzzle = $this->guzzle;

                    try {
                        $echoToken = $this->echo->token();
                    } catch(Exception $e){
                        response(500, "System configuration error!");
                    }

                    $account = $this->system->getWaAccount($api["uid"], $request["account"], "id");

                    $filtered = [
                        "uid" => $api["uid"],
                        "wid" => $account["wid"],
                        "phone" => $request["phone"],
                        "message" => bjoernffm\Spintax\Parser::parse(footermark($subscription["footermark"], $request["message"], system_message_mark))->generate(),
                        "status" => 1,
                        "api" => 1,
                        "create_date" => date("Y-m-d H:i:s", time())
                    ];

                    if($this->system->create("wa_sent", $filtered)):
                        if($echoToken):
                            $this->wa->_guzzle = $this->guzzle;

                            if($this->wa->send($echoToken, $account["unique"], $api["uid"], $api["hash"])):
                                $this->echo->notify($api["hash"], [
                                    "type" => "table"
                                ]);
                            else:
                                response(500, "Unable to connect to WhatsApp servers!");
                            endif;
                        else:
                            response(500, "Something went wrong!");
                        endif;
                    else:
                        response(500, "Something went wrong!");
                    endif;
                endif;

                response(200, "OTP has been sent!", [
                    "phone" => $request["phone"],
                    "message" => $request["message"],
                    "otp" => (int) $otp
                ]);

                break;
            case "sms":
                /**
                 * @api {post} /send/sms Send Single Message
                 * @apiName Send Single Message
                 * @apiDescription Send a single sms message. Requires "<strong>sms_send</strong>" API permission.
                 * @apiGroup SMS
                 * @apiVersion 1.0.0
                 *
                 * @apiParam {String} secret The API secret you copied from (Tools -> API Keys) page
                 * @apiParam {String="devices","credits"} mode Mode of sending the message, it can be "devices" which will allow you to use your linked android devices or "credits" which will allow you to use gateways and partner devices. "credits" requires you to have enough credit balance to send messages.
                 * @apiParam {String} phone Recipient mobile number, it will accept E.164 formatted number or locally formatted numbers using the country code from your profile settings.<br>
                 * <strong>Example for Philippines</strong><br>
                 * E.164: +639184661533<br>
                 * Local: 09184661533
                 * @apiParam {String} message Message you want to send, spintax is also supported.
                 * @apiParam {String} [device] Linked device unique ID, this is required if you will send with "devices" mode. You can get linked device unique ID from <strong>/get/devices</strong> (Your devices).
                 * @apiParam {String|Number} [gateway] Partner device unique ID or gateway ID, this is required if you will send with "credits" mode. You can get a partner device unique ID and gateway ID from <strong>/get/rates</strong> 
                 * @apiParam {Number=1,2} [sim] Sim slot number you want to use. For "devices" mode only.
                 * @apiParam {Number=1,2} [priority=1] If you want to send the messages as priority, 1 for yes and 2 for no. For "devices" mode only.
                 * @apiParam {Number} [shortener=none] Shortener ID, specify the shortener you want to use if you want to shorten the links in your message. You can get the list of available shorteners from <strong>/get/shorteners</strong>
                 *
                 * @apiExample {php} PHP Example
                  <?php

                    $message = [
                        "secret" => "API_SECRET", // your API secret from (Tools -> API Keys) page
                        "mode" => "devices",
                        "device" => "00000000-0000-0000-d57d-f30cb6a89289",
                        "sim" => 1,
                        "priority" => 1,
                        "phone" => "+639123456789",
                        "message" => "Hello World!"
                    ];

                    $cURL = curl_init("http://127.0.0.1/zender/api/send/sms");
                    curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    curl_setopt($cURL, CURLOPT_POSTFIELDS, $message);
                    $response = curl_exec($cURL);
                    curl_close($cURL);

                    $result = json_decode($response, true);

                    // do something with response
                    print_r($result);
                 * 
                 * @apiExample {python} Python Example
                    import requests
    
                    # your API secret from (Tools -> API Keys) page
                    apiSecret = "API_SECRET"

                    message = {
                        "secret": apiSecret,
                        "mode": "devices",
                        "device": "00000000-0000-0000-d57d-f30cb6a89289",
                        "sim": 1,
                        "priority": 1,
                        "phone": "+639123456789",
                        "message": "Hello World!"
                    }

                    r = requests.post(url = "http://127.0.0.1/zender/api/send/sms", params = message)
                      
                    # do something with response object
                    result = r.json()
                 *
                 * @apiSuccess (Success Response Format) {Number} status List of Codes
                 * <br> 200 = Success
                 * @apiSuccess (Success Response Format) {String} message Response message
                 * @apiSuccess (Success Response Format) {Array} data Array of data
                 *
                 * @apiSuccessExample {json} Success Response
                 {
                   "status": 200,
                   "message": "Message has been queued for sending!",
                   "data": false
                 }
                 * 
                 * @apiError (Error Response Format) {Number} status List of Codes<br>
                 * 400 = Invalid parameters<br>
                 * 401 = Invalid API secret<br>
                 * 403 = Access denied<br>
                 * 404 = Device doesn't exist<br>
                 * 500 = Something went wrong
                 * @apiError (Error Response Format) {String} message Response message
                 * @apiError (Error Response Format) {Array} data Array of data
                 * 
                 * @apiErrorExample {json} Error Response
                 {
                   "status": 400,
                   "message": "Invalid Parameters!",
                   "data": false
                 }
                 *
                 */

                if(!in_array("sms_send", explode(",", $api["permissions"])))
                    response(403, "This API key doesn't have permission to use this endpoint!");

                if(!isset($request["mode"], $request["phone"], $request["message"]))
                    response(400, "Invalid Parameters!");

                if(!$this->file->exists("system/storage/temporary/firebase.json")):
                    response(500, "System configuration error!");
                endif;

                try {
                    $number = $this->phone->parse($request["phone"], $api["country"]);

                    $number->format(Brick\PhoneNumber\PhoneNumberFormat::INTERNATIONAL);

                    if(!$number->isValidNumber())
                        response(400, "Invalid phone number!");

                    if(!$number->getNumberType(Brick\PhoneNumber\PhoneNumberType::MOBILE))
                        response(400, "Invalid phone number!");

                    $phone = $number->format(Brick\PhoneNumber\PhoneNumberFormat::E164);
                    $country = $number->getRegionCode();
                } catch(Brick\PhoneNumber\PhoneNumberParseException $e) {
                    response(400, "Invalid phone number!");
                }

                if(!$this->sanitize->length($request["message"], system_message_min))
                    response(400, "Message is too short!");

                if(system_message_max > 0):
                    if($this->sanitize->length($request["message"], system_message_max, 2))
                        response(400, "Message is too long!");
                endif;

                if($request["mode"] == "devices"):
                    if(!isset(
                        $request["device"],
                        $request["sim"]
                    ))
                        response(400, "Invalid Parameters!");

                    if(!isset($request["priority"]))
                        $request["priority"] = 1;

                    $subscription = set_subscription(
                        $this->system->checkSubscription($api["uid"]), 
                        $this->system->getSubscription(false, $api["uid"]), 
                        $this->system->getSubscription(false, false, true)
                    );

                    if(empty($subscription))
                        response(403, "Account is not subscribed to any premium package!");

                    if(!$this->sanitize->isInt($request["sim"]))
                        response(400, "Invalid Parameters!");

                    if(!$this->sanitize->isInt($request["priority"]))
                        response(400, "Invalid Parameters!");

                    if(limitation($subscription["send_limit"], $this->system->countQuota($api["uid"], "sent")))
                        response(403, "Maximum allowed number of sent messages has been reached!");

                    if($this->system->checkDevice($api["uid"], $request["device"], "did") < 1)
                        response(404, "Device doesn't exist!");

                    $device = $this->system->getDevice($api["uid"], $request["device"], "did");
                else:
                    if($request["mode"] != "credits")
                        response(400, "Invalid Parameters!");

                    if(!isset($request["gateway"]))
                        response(400, "Invalid Parameters!");

                    if($this->sanitize->isInt($request["gateway"])):
                        $gateways = $this->system->getGateways();

                        if(!array_key_exists($request["gateway"], $gateways)):
                            response(404, "Specified gateway doesn't exist!");
                        endif;

                        if(!$this->file->exists("system/gateways/" . md5($request["gateway"]) . ".php"))
                            response(404, "Specified gateway doesn't exist!");

                        require "system/gateways/" . md5($request["gateway"]) . ".php";
                    else:
                        $device = $this->system->getDevice(false, $request["gateway"], "global");

                        if($device):
                            if($device["global_device"] > 1):
                                response(403, "Device is not available!");
                            endif;
                        else:
                            response(404, "Device doesn't exist!");
                        endif;
                    endif;
                endif;

                if(isset($request["shortener"])):
                    if(!$this->sanitize->isInt($request["shortener"]))
                        response(400, "Invalid Parameters!");

                    if($request["shortener"] > 0):
                        if(!$this->file->exists("system/shorteners/" . md5($request["shortener"]) . ".php"))
                            response(404, "Specified shortener doesn't exist!");

                        $messageLinks = (new VStelmakh\UrlHighlight\UrlHighlight)->getUrls($request["message"]);

                        if(!empty($messageLinks)):
                            require "system/shorteners/" . md5($request["shortener"]) . ".php";

                            foreach($messageLinks as $key => $value):
                                $shortLink = shortenUrl($value, $this);

                                if($shortLink):
                                    $request["message"] = str_replace("{$value}", "{$shortLink}", $request["message"]);
                                endif;
                            endforeach;
                        endif;
                    endif;
                endif;

                if($request["mode"] == "devices"):
                    $this->system->create("sent", [
                        "uid" => $api["uid"],
                        "did" => $request["device"],
                        "gateway" => 0,
                        "sim" => $request["sim"] < 2 ? 1 : 2,
                        "mode" => 1,
                        "phone" => $phone,
                        "message" => bjoernffm\Spintax\Parser::parse(footermark($subscription["footermark"], $request["message"], system_message_mark))->generate(),
                        "status" => 1,
                        "status_code" => false,
                        "priority" => $request["priority"] < 2 ? 1 : 2,
                        "api" => 1,
                        "create_date" => date("Y-m-d H:i:s", time())
                    ]);
                else:
                    $credits = $this->system->getCredits($api["uid"]);

                    if($this->sanitize->isInt($request["gateway"])):
                        $pricing = json_decode($gateways[$request["gateway"]]["pricing"], true);

                        if(array_key_exists(strtolower($country), $pricing["countries"])):
                            $price = $pricing["countries"][strtolower($country)];
                        else:
                            $price = $pricing["default"];
                        endif;

                        if($credits < $price)
                            response(403, "Not enough credits to send this message!");

                        $gateway = $gateways[$request["gateway"]];

                        $message = bjoernffm\Spintax\Parser::parse($request["message"])->generate();

                        $send = gatewaySend($request["phone"], $message, $this);

                        if($send):
                            $create = $this->system->create("sent", [
                                "uid" => $api["uid"],
                                "did" => false,
                                "gateway" => $request["gateway"],
                                "api" => 1,
                                "sim" => 0,
                                "mode" => 2,
                                "priority" => 0,
                                "phone" => $phone,
                                "message" => $message,
                                "status" => $gateway["callback"] < 2 ? 2 : 3,
                                "status_code" => false
                            ]);

                            if($gateway["callback"] < 2):
                                $this->cache->container("system.gateways");

                                $this->cache->set("{$gateway["callback_id"]}.{$send}", $create);

                                response(200, "Message has been queued for sending!");
                            else:
                                response(200, "Message has been sent!");
                            endif;
                        else:
                            response(500, "Gateway was unable to send your message!");
                        endif;
                    else:
                        $currency = country($device["country"])->getCurrency()["iso_4217_code"];

                        $this->cache->container("system.payments", true);

                        if(!$this->cache->has("exchange")):
                            try {
                                $exchange = json_decode($this->guzzle->get(titansys_api . "/currency?code=" . system_purchase_code, [
                                    "allow_redirects" => true,
                                    "http_errors" => false
                                ])->getBody()->getContents(), true);

                                if($exchange["status"] == 200):
                                    $this->cache->set("exchange", $exchange, 43200);
                                else:
                                    response(500, "System configuration error!");
                                endif;
                            } catch(Exception $e){
                                response(500, "System configuration error!");
                            }
                        endif;

                        $rates = $this->cache->get("exchange");

                        $base_rate = $rates["data"]["USD"] / $rates["data"][strtoupper($currency)];
                        $usd_price = ($base_rate * $device["rate"]) * $rates["data"]["USD"];
                        $final_price = (float) abs($usd_price * $rates["data"][strtoupper(system_currency)]);

                        if($credits < $final_price)
                            response(403, "Not enough credits to send this message!");

                        $slots = explode(",", $device["global_slots"]);

                        $this->system->create("sent", [
                            "uid" => $api["uid"],
                            "did" => $request["gateway"],
                            "gateway" => 0,
                            "sim" => count($slots) > 1 ? rand(1, 2) : ($slots[0] < 2 ? 1 : 2),
                            "mode" => 2,
                            "phone" => $phone,
                            "message" => bjoernffm\Spintax\Parser::parse($request["message"])->generate(),
                            "status" => 1,
                            "status_code" => false,
                            "priority" => $device["global_priority"],
                            "api" => 1,
                            "create_date" => date("Y-m-d H:i:s", time())
                        ]);
                    endif;
                endif;

                $this->echo->_cache = $this->cache;
                $this->echo->_guzzle = $this->guzzle;

                try {
                    $echoToken = $this->echo->token();

                    if($echoToken):
                        $this->echo->notify($api["hash"], [
                            "type" => "table"
                        ]);
                    endif;
                } catch(Exception $e){
                    // Ignore
                }

                if($request["mode"] == "devices"):
                    if(round((int) $device["version"]) > 7):
                        $this->fcm->send(md5($api["uid"] . $request["device"]), [
                            "type" => "sms",
                            "global" => 0,
                            "currency" => "None",
                            "rate" => (float) 0
                        ]);
                    else:
                        try {
                            $echoToken = $this->echo->token();

                            if($echoToken):
                                $this->echo->notify("legacy_" . md5($api["uid"] . $request["device"]), [
                                    "type" => "sms",
                                    "global" => 0,
                                    "currency" => "None",
                                    "rate" => (float) 0
                                ]);
                            endif;
                        } catch(Exception $e){
                            // Ignore;
                        }
                    endif;
                else:
                    if(!$this->sanitize->isInt($request["gateway"])):
                        if(round((int) $device["version"]) > 7):
                            $this->fcm->send(md5($device["uid"] . $request["gateway"]), [
                                "type" => "sms",
                                "global" => 1,
                                "currency" => $currency,
                                "rate" => (float) $device["rate"]
                            ]);
                        else:
                            try {
                                $echoToken = $this->echo->token();

                                if($echoToken):
                                    $this->echo->notify("legacy_" . md5($device["uid"] . $request["gateway"]), [
                                        "type" => "sms",
                                        "global" => 1,
                                        "currency" => $currency,
                                        "rate" => (float) $device["rate"]
                                    ]);
                                endif;
                            } catch(Exception $e){
                                // Ignore
                            }
                        endif;
                    endif;
                endif;

                response(200, "Message has been queued for sending!");

                break;
            case "sms.bulk":
                /**
                 * @api {post} /send/sms.bulk Send Bulk Messages
                 * @apiName Send Bulk Message
                 * @apiDescription Send bulk sms messages. Requires "<strong>sms_send_bulk</strong>" API permission.
                 * @apiGroup SMS
                 * @apiVersion 1.0.0
                 *
                 * @apiParam {String} secret The API secret you copied from (Tools -> API Keys) page
                 * @apiParam {String="devices","credits"} mode Mode of sending the message, it can be "devices" which will allow you to use your linked android devices or "credits" which will allow you to use gateways and partner devices. "credits" requires you to have enough credit balance to send messages.
                 * @apiParam {String} [numbers] List of phone numbers separated by commas. It can be optional if "groups" parameter is not empty. It will accept E.164 formatted number or locally formatted numbers using the country code from your profile settings.<br>
                 * <strong>Example for Philippines</strong><br>
                 * E.164: +639184661533<br>
                 * Local: 09184661533
                 * @apiParam {String} [groups] List of contact group ID's separated by commas. It can be optional if "numbers" parameter is not empty. You can get group ID's from <strong>/get/groups</strong> (Your contact groups).
                 * @apiParam {String} message Message you want to send, spintax and shortcodes are supported.
                 * @apiParam {String} [device] Linked device unique ID, this is required if you will send with "devices" mode. You can get linked device unique ID from <strong>/get/devices</strong> (Your devices).
                 * @apiParam {String|Number} [gateway] Partner device unique ID or gateway ID, this is required if you will send with "credits" mode. You can get a partner device unique ID and gateway ID from <strong>/get/rates</strong> 
                 * @apiParam {Number=1,2} [sim] Sim slot number you want to use. For "devices" mode only.
                 * @apiParam {Number=1,2} [priority=1] If you want to send the messages as priority, 1 for yes and 2 for no. For "devices" mode only.
                 * @apiParam {Number} [shortener=none] Shortener ID, specify the shortener you want to use if you want to shorten the links in your message. You can get the list of available shorteners from <strong>/get/shorteners</strong>
                 *
                 * @apiExample {php} PHP Example
                  <?php

                    $message = [
                        "secret" => "API_SECRET", // your API secret from (Tools -> API Keys) page
                        "mode" => "devices",
                        "numbers" => "+639123456789,+639123456789,+639123456789",
                        "groups" => "1,2,3,4",
                        "device" => "00000000-0000-0000-d57d-f30cb6a89289",
                        "sim" => 1,
                        "priority" => 1,
                        "message" => "Hello World!"
                    ];

                    $cURL = curl_init("http://127.0.0.1/zender/api/send/sms.bulk");
                    curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    curl_setopt($cURL, CURLOPT_POSTFIELDS, $message);
                    $response = curl_exec($cURL);
                    curl_close($cURL);

                    $result = json_decode($response, true);

                    // do something with response
                    print_r($result);
                 * 
                 * @apiExample {python} Python Example
                    import requests
    
                    # your API secret from (Tools -> API Keys) page
                    apiSecret = "API_SECRET"

                    message = {
                        "secret": apiSecret,
                        "mode": "devices",
                        "numbers": "+639123456789,+639123456789,+639123456789",
                        "groups": "1,2,3,4",
                        "device": "00000000-0000-0000-d57d-f30cb6a89289",
                        "sim": 1,
                        "priority": 1,
                        "message": "Hello World!"
                    }

                    r = requests.post(url = "http://127.0.0.1/zender/api/send/sms.bulk", params = message)
                      
                    # do something with response object
                    result = r.json()
                 *
                 * @apiSuccess (Success Response Format) {Number} status List of Codes
                 * <br> 200 = Success
                 * @apiSuccess (Success Response Format) {String} message Response message
                 * @apiSuccess (Success Response Format) {Array} data Array of data
                 *
                 * @apiSuccessExample {json} Success Response
                 {
                   "status": 200,
                   "message": "Message has been queued for sending!",
                   "data": false
                 }
                 * 
                 * @apiError (Error Response Format) {Number} status List of Codes<br>
                 * 400 = Invalid parameters<br>
                 * 401 = Invalid API secret<br>
                 * 403 = Access denied<br>
                 * 404 = Device doesn't exist<br>
                 * 500 = Something went wrong
                 * @apiError (Error Response Format) {String} message Response message
                 * @apiError (Error Response Format) {Array} data Array of data
                 * 
                 * @apiErrorExample {json} Error Response
                 {
                   "status": 400,
                   "message": "Invalid Parameters!",
                   "data": false
                 }
                 *
                 */

                if(!in_array("sms_send_bulk", explode(",", $api["permissions"])))
                    response(403, "This API key doesn't have permission to use this endpoint!");

                if(!isset($request["mode"], $request["message"]))
                    response(400, "Invalid Parameters!");

                if(!isset($request["numbers"]) && !isset($request["groups"]))
                    response(400, "Invalid Parameters!");

                if(!$this->file->exists("system/storage/temporary/firebase.json")):
                    response(500, "System configuration error!");
                endif;

                if(!$this->sanitize->length($request["message"], system_message_min))
                    response(400, "Message is too short!");

                if(system_message_max > 0):
                    if($this->sanitize->length($request["message"], system_message_max, 2))
                        response(400, "Message is too long!");
                endif;

                if($request["mode"] == "devices"):
                    if(!isset(
                        $request["device"],
                        $request["sim"]
                    ))
                        response(400, "Invalid Parameters!");

                    if(!isset($request["priority"]))
                        $request["priority"] = 2;

                    $subscription = set_subscription(
                        $this->system->checkSubscription($api["uid"]), 
                        $this->system->getSubscription(false, $api["uid"]), 
                        $this->system->getSubscription(false, false, true)
                    );

                    if(empty($subscription))
                        response(403, "You are not subscribed to any premium package!");

                    if(!$this->sanitize->isInt($request["sim"]))
                        response(400, "Invalid Parameters!");

                    if(!$this->sanitize->isInt($request["priority"]))
                        response(400, "Invalid Parameters!");

                    if($this->system->checkDevice($api["uid"], $request["device"], "did") < 1)
                        response(404, "Device doesn't exist!");

                    $device = $this->system->getDevice($api["uid"], $request["device"], "did");
                else:
                    if($request["mode"] != "credits")
                        response(400, "Invalid Parameters!");

                    if(!isset($request["gateway"]))
                        response(400, "Invalid Parameters!");

                    if($this->sanitize->isInt($request["gateway"])):
                        $gateways = $this->system->getGateways();

                        if(!array_key_exists($request["gateway"], $gateways)):
                            response(404, "Specified gateway doesn't exist!");
                        endif;

                        if(!$this->file->exists("system/gateways/" . md5($request["gateway"]) . ".php"))
                            response(404, "Specified gateway doesn't exist!");

                        require "system/gateways/" . md5($request["gateway"]) . ".php";
                    else:
                        $device = $this->system->getDevice(false, $request["gateway"], "global");

                        if($device):
                            if($device["global_device"] > 1):
                                response(403, "Device is not available!");
                            endif;
                        else:
                            response(404, "Device doesn't exist!");
                        endif;
                    endif;
                endif;

                $contactBook = [];

                if(isset($request["numbers"])):
                    $numbers = explode(",", trim($request["numbers"]));

                    if(is_array($numbers) && !empty($numbers) && !empty($numbers[0])):
                        foreach($numbers as $number):
                            $rejected = false;

                            try {
                                $phone = $this->phone->parse($number, $api["country"]);

                                $phone->format(Brick\PhoneNumber\PhoneNumberFormat::INTERNATIONAL);

                                if(!$phone->isValidNumber())
                                    $rejected = true;

                                if(!$phone->getNumberType(Brick\PhoneNumber\PhoneNumberType::MOBILE))
                                    $rejected = true;

                                $phoneNumber = $phone->format(Brick\PhoneNumber\PhoneNumberFormat::E164);
                                $country = $phone->getRegionCode();
                            } catch(Brick\PhoneNumber\PhoneNumberParseException $e) {
                                $rejected = true;
                            }

                            if(!$rejected):
                                $contactBook[] = [
                                    "name" => $phoneNumber,
                                    "phone" => $phoneNumber,
                                    "group" => "Unknown",
                                    "country" => $country
                                ];
                            endif;
                        endforeach;
                    endif;
                endif;

                if(isset($request["groups"])):
                    $groups = explode(",", trim($request["groups"]));

                    if(is_array($groups) && !empty($groups) && !empty($groups[0])):
                        foreach($groups as $group):
                            if($this->system->checkGroup($api["uid"], $group) > 0):
                                $contacts = $this->system->getContactsByGroup($api["uid"], $group);

                                if(!empty($contacts)):
                                    foreach($contacts as $contact):
                                        try {
                                            $phone = $this->phone->parse($contact["phone"], $api["country"]);
                                            $country = $phone->getRegionCode();
                                        } catch(Brick\PhoneNumber\PhoneNumberParseException $e) {
                                            // ignore
                                        }

                                        $contactBook[] = [
                                            "name" => $contact["name"],
                                            "phone" => $contact["phone"],
                                            "group" => $contact["group"],
                                            "country" => $country
                                        ];
                                    endforeach;
                                endif;
                            endif;
                        endforeach;
                    endif;
                endif;

                if(empty($contactBook))
                    response(400, "Invalid Parameters!");

                if(isset($request["shortener"])):
                    if(!$this->sanitize->isInt($request["shortener"]))
                        response(400, "Invalid Parameters!");

                    if($request["shortener"] > 0):
                        if(!$this->file->exists("system/shorteners/" . md5($request["shortener"]) . ".php"))
                            response(404, "Specified shortener doesn't exist!");

                        $messageLinks = (new VStelmakh\UrlHighlight\UrlHighlight)->getUrls($request["message"]);

                        if(!empty($messageLinks)):
                            require "system/shorteners/" . md5($request["shortener"]) . ".php";

                            foreach($messageLinks as $key => $value):
                                $shortLink = shortenUrl($value, $this);

                                if($shortLink):
                                    $request["message"] = str_replace("{$value}", "{$shortLink}", $request["message"]);
                                endif;
                            endforeach;
                        endif;
                    endif;
                endif;

                foreach($contactBook as $contact):
                    if($request["mode"] == "devices"):
                        if(!limitation($subscription["send_limit"], $this->system->countQuota($api["uid"], "sent"))):
                            $this->system->create("sent", [
                                "uid" => $api["uid"],
                                "did" => $request["device"],
                                "gateway" => 0,
                                "sim" => $request["sim"] < 2 ? 1 : 2,
                                "mode" => 1,
                                "phone" => $contact["phone"],
                                "message" => bjoernffm\Spintax\Parser::parse($this->lex->parse(footermark($subscription["footermark"], $request["message"], system_message_mark), [
                                    "contact" => [
                                        "name" => $contact["name"],
                                        "number" => $contact["phone"]
                                    ],
                                    "group" => [
                                        "name" => $contact["group"]
                                    ],
                                    "date" => [
                                        "now" => date("F j, Y"),
                                        "time" => date("h:i A") 
                                    ]
                                ]))->generate(),
                                "status" => 1,
                                "status_code" => false,
                                "priority" => $request["priority"] < 1 ? 1 : 2,
                                "api" => 1,
                                "create_date" => date("Y-m-d H:i:s", time())
                            ]);
                        endif;
                    else:
                        $credits = $this->system->getCredits($api["uid"]);

                        if($this->sanitize->isInt($request["gateway"])):
                            $pricing = json_decode($gateways[$request["gateway"]]["pricing"], true);

                            if(array_key_exists(strtolower($contact["country"]), $pricing["countries"])):
                                $price = $pricing["countries"][strtolower($contact["country"])];
                            else:
                                $price = $pricing["default"];
                            endif;

                            if($credits >= ($price * count($contactBook))):
                                $gateway = $gateways[$request["gateway"]];

                                $message = bjoernffm\Spintax\Parser::parse($request["message"])->generate();

                                $send = gatewaySend($contact["phone"], $message, $this);

                                if($send):
                                    $create = $this->system->create("sent", [
                                        "uid" => $api["uid"],
                                        "did" => false,
                                        "gateway" => $request["gateway"],
                                        "api" => 1,
                                        "sim" => 0,
                                        "mode" => 2,
                                        "priority" => 0,
                                        "phone" => $contact["phone"],
                                        "message" => $message,
                                        "status" => $gateway["callback"] < 2 ? 2 : 3,
                                        "status_code" => false
                                    ]);

                                    if($gateway["callback"] < 2):
                                        $this->cache->container("system.gateways");

                                        $this->cache->set("{$gateway["callback_id"]}.{$send}", $create);
                                    endif;
                                endif;
                            endif;
                        else:
                            $currency = country($device["country"])->getCurrency()["iso_4217_code"];

                            $this->cache->container("system.payments", true);

                            if(!$this->cache->has("exchange")):
                                try {
                                    $exchange = json_decode($this->guzzle->get(titansys_api . "/currency?code=" . system_purchase_code, [
                                        "allow_redirects" => true,
                                        "http_errors" => false
                                    ])->getBody()->getContents(), true);

                                    if($exchange["status"] == 200):
                                        $this->cache->set("exchange", $exchange, 43200);
                                    else:
                                        response(500, "System configuration error!");
                                    endif;
                                } catch(Exception $e){
                                    response(500, "System configuration error!");
                                }
                            endif;

                            $rates = $this->cache->get("exchange");

                            $base_rate = $rates["data"]["USD"] / $rates["data"][strtoupper($currency)];
                            $usd_price = ($base_rate * $device["rate"]) * $rates["data"]["USD"];
                            $final_price = (float) abs($usd_price * $rates["data"][strtoupper(system_currency)]);

                            if($credits >= ($final_price * count($contactBook))):
                                $slots = explode(",", $device["global_slots"]);

                                $sim = count($slots) > 1 ? rand(1, 2) : ($slots[0] < 2 ? 1 : 2);

                                $this->system->create("sent", [
                                    "uid" => $api["uid"],
                                    "did" => $request["gateway"],
                                    "gateway" => 0,
                                    "sim" => $sim,
                                    "mode" => 2,
                                    "phone" => $contact["phone"],
                                    "message" => bjoernffm\Spintax\Parser::parse($this->lex->parse($request["message"], [
                                        "contact" => [
                                            "name" => $contact["name"],
                                            "number" => $contact["phone"]
                                        ],
                                        "group" => [
                                            "name" => $contact["group"]
                                        ],
                                        "date" => [
                                            "now" => date("F j, Y"),
                                            "time" => date("h:i A") 
                                        ]
                                    ]))->generate(),
                                    "status" => 1,
                                    "status_code" => false,
                                    "priority" => $request["priority"] < 1 ? 1 : 2,
                                    "api" => 1,
                                    "create_date" => date("Y-m-d H:i:s", time())
                                ]);
                            endif;
                        endif;
                    endif;
                endforeach;

                $this->echo->_cache = $this->cache;
                $this->echo->_guzzle = $this->guzzle;

                try {
                    $echoToken = $this->echo->token();

                    if($echoToken):
                        $this->echo->notify($api["hash"], [
                            "type" => "table"
                        ]);
                    endif;
                } catch(Exception $e){
                    // Ignore
                }

                if($request["mode"] == "devices"):
                    if(round((int) $device["version"]) > 7):
                        $this->fcm->send(md5($api["uid"] . $request["device"]), [
                            "type" => "sms",
                            "global" => 0,
                            "currency" => "None",
                            "rate" => (float) 0
                        ]);
                    else:
                        try {
                            $echoToken = $this->echo->token();

                            if($echoToken):
                                $this->echo->notify("legacy_" . md5($api["uid"] . $request["device"]), [
                                    "type" => "sms",
                                    "global" => 0,
                                    "currency" => "None",
                                    "rate" => (float) 0
                                ]);
                            endif;
                        } catch(Exception $e){
                            // Ignore
                        }
                    endif;
                else:
                    if(!$this->sanitize->isInt($request["gateway"])):
                        if(round((int) $device["version"]) > 7):
                            $this->fcm->send(md5($device["uid"] . $request["gateway"]), [
                                "type" => "sms",
                                "global" => 1,
                                "currency" => $currency,
                                "rate" => (float) $device["rate"]
                            ]);
                        else:
                            try {
                                $echoToken = $this->echo->token();

                                if($echoToken):
                                    $this->echo->notify("legacy_" . md5($device["uid"] . $request["gateway"]), [
                                        "type" => "sms",
                                        "global" => 1,
                                        "currency" => $currency,
                                        "rate" => (float) $device["rate"]
                                    ]);
                                endif;
                            } catch(Exception $e){
                                // Ignore
                            }
                        endif;
                    endif;
                endif;

                response(200, "Bulk messages has been queued for sending!");

                break;
            case "whatsapp":
                /**
                 * @api {post} /send/whatsapp Send Single Chat
                 * @apiName Send Single Chat
                 * @apiDescription Send a single chat message. Requires "<strong>wa_send</strong>" API permission.
                 * @apiGroup WhatsApp
                 * @apiVersion 1.0.0
                 *
                 * @apiParam {String} secret The API secret you copied from (Tools -> API Keys) page
                 * @apiParam {Number} account WhatsApp account ID you want to use for sending, you can get account ID's from <strong>/get/wa.accounts</strong> (Your WhatsApp accounts).
                 * @apiParam {String} phone Recipient mobile number, it will accept E.164 formatted number or locally formatted numbers using the country code from your profile settings.<br>
                 * <strong>Example for Philippines</strong><br>
                 * E.164: +639184661533<br>
                 * Local: 09184661533
                 * @apiParam {String} message Message you want to send, spintax is also supported.
                 * @apiParam {Number} [shortener=none] Shortener ID, specify the shortener you want to use if you want to shorten the links in your message. You can get the list of available shorteners from <strong>/get/shorteners</strong>
                 *
                 * @apiExample {php} PHP Example
                  <?php

                    $chat = [
                        "secret" => "API_SECRET", // your API secret from (Tools -> API Keys) page
                        "account" => 1,
                        "phone" => "+639123456789",
                        "message" => "Hello World!"
                    ];

                    $cURL = curl_init("http://127.0.0.1/zender/api/send/whatsapp");
                    curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    curl_setopt($cURL, CURLOPT_POSTFIELDS, $chat);
                    $response = curl_exec($cURL);
                    curl_close($cURL);

                    $result = json_decode($response, true);

                    // do something with response
                    print_r($result);
                 * 
                 * @apiExample {python} Python Example
                    import requests
    
                    # your API secret from (Tools -> API Keys) page
                    apiSecret = "API_SECRET"

                    chat = {
                        "secret": apiSecret,
                        "account": 1,
                        "phone": "+639123456789",
                        "message": "Hello World!"
                    }

                    r = requests.post(url = "http://127.0.0.1/zender/api/send/whatsapp", params = chat)
                      
                    # do something with response object
                    result = r.json()
                 *
                 * @apiSuccess (Success Response Format) {Number} status List of Codes
                 * <br> 200 = Success
                 * @apiSuccess (Success Response Format) {String} message Response message
                 * @apiSuccess (Success Response Format) {Array} data Array of data
                 *
                 * @apiSuccessExample {json} Success Response
                 {
                   "status": 200,
                   "message": "WhatsApp message has been queued for sending!",
                   "data": false
                 }
                 * 
                 * @apiError (Error Response Format) {Number} status List of Codes<br>
                 * 400 = Invalid parameters<br>
                 * 401 = Invalid API secret<br>
                 * 403 = Access denied<br>
                 * 404 = WhatsApp account doesn't exist<br>
                 * 500 = Something went wrong
                 * @apiError (Error Response Format) {String} message Response message
                 * @apiError (Error Response Format) {Array} data Array of data
                 * 
                 * @apiErrorExample {json} Error Response
                 {
                   "status": 400,
                   "message": "Invalid Parameters!",
                   "data": false
                 }
                 *
                 */

                if(!in_array("wa_send", explode(",", $api["permissions"])))
                    response(403, "This API key doesn't have permission to use this endpoint!");

                if(!isset($request["phone"], $request["account"], $request["message"]))
                    response(400, "Invalid Parameters!");

                $subscription = set_subscription(
                    $this->system->checkSubscription($api["uid"]), 
                    $this->system->getSubscription(false, $api["uid"]), 
                    $this->system->getSubscription(false, false, true)
                );

                if(empty($subscription))
                    response(403, "Account is not subscribed to any premium package!");

                if($this->system->checkWaAccount($api["uid"], $request["account"], "id") < 1)
                    response(404, "WhatsApp account doesn't exist!");

                try {
                    $number = $this->phone->parse($request["phone"], $api["country"]);

                    $number->format(Brick\PhoneNumber\PhoneNumberFormat::INTERNATIONAL);

                    if(!$number->isValidNumber())
                        response(400, "Invalid phone number!");

                    if(!$number->getNumberType(Brick\PhoneNumber\PhoneNumberType::MOBILE))
                        response(400, "Invalid phone number!");

                    $request["phone"] = $number->format(Brick\PhoneNumber\PhoneNumberFormat::E164);
                } catch(Brick\PhoneNumber\PhoneNumberParseException $e) {
                    response(400, "Invalid phone number!");
                }

                if(!$this->sanitize->length($request["message"], system_message_min))
                    response(400, "Chat message is too short!");

                if(system_message_max > 0):
                    if($this->sanitize->length($request["message"], system_message_max, 2))
                        response(400, "Chat message is too long!");
                endif;

                if(limitation($subscription["wa_send_limit"], $this->system->countQuota($api["uid"], "wa_sent")))
                    response(403, "You have reached the maximum number of allowed chats!");

                if(isset($request["shortener"])):
                    if(!$this->sanitize->isInt($request["shortener"]))
                        response(400, "Invalid Parameters!");

                    if($request["shortener"] > 0):
                        if(!$this->file->exists("system/shorteners/" . md5($request["shortener"]) . ".php"))
                            response(404, "Specified shortener doesn't exist!");

                        $messageLinks = (new VStelmakh\UrlHighlight\UrlHighlight)->getUrls($request["message"]);

                        if(!empty($messageLinks)):
                            require "system/shorteners/" . md5($request["shortener"]) . ".php";

                            foreach($messageLinks as $key => $value):
                                $shortLink = shortenUrl($value, $this);

                                if($shortLink):
                                    $request["message"] = str_replace("{$value}", "{$shortLink}", $request["message"]);
                                endif;
                            endforeach;
                        endif;
                    endif;
                endif;

                $this->echo->_cache = $this->cache;
                $this->echo->_guzzle = $this->guzzle;

                try {
                    $echoToken = $this->echo->token();
                } catch(Exception $e){
                    response(500, "System configuration error!");
                }

                $account = $this->system->getWaAccount($api["uid"], $request["account"], "id");

                $filtered = [
                    "uid" => $api["uid"],
                    "wid" => $account["wid"],
                    "phone" => $request["phone"],
                    "message" => bjoernffm\Spintax\Parser::parse(footermark($subscription["footermark"], $request["message"], system_message_mark))->generate(),
                    "status" => 1,
                    "api" => 1,
                    "create_date" => date("Y-m-d H:i:s", time())
                ];

                if($this->system->create("wa_sent", $filtered)):
                    if($echoToken):
                        $this->wa->_guzzle = $this->guzzle;

                        if($this->wa->send($echoToken, $account["unique"], $api["uid"], $api["hash"])):
                            $this->echo->notify($api["hash"], [
                                "type" => "table"
                            ]);

                            response(200, "WhatsApp chat has been queued for sending!");
                        else:
                            response(500, "Unable to connect to WhatsApp servers!");
                        endif;
                    else:
                        response(500, "Something went wrong!");
                    endif;
                else:
                    response(500, "Something went wrong!");
                endif;

                break;
            case "whatsapp.bulk":
                /**
                 * @api {post} /send/whatsapp.bulk Send Bulk Chats
                 * @apiName Send Bulk Chats
                 * @apiDescription Send bulk chat messages. Requires "<strong>wa_send_bulk</strong>" API permission.
                 * @apiGroup WhatsApp
                 * @apiVersion 1.0.0
                 *
                 * @apiParam {String} secret The API secret you copied from (Tools -> API Keys) page
                 * @apiParam {Number} account WhatsApp account ID you want to use for sending, you can get account ID's from <strong>/get/wa.accounts</strong> (Your WhatsApp accounts).
                 * @apiParam {String} [numbers] List of phone numbers separated by commas. It can be optional if "groups" parameter is not empty. It will accept E.164 formatted number or locally formatted numbers using the country code from your profile settings.<br>
                 * <strong>Example for Philippines</strong><br>
                 * E.164: +639184661533<br>
                 * Local: 09184661533
                 * @apiParam {String} [groups] List of contact group ID's separated by commas. It can be optional if "numbers" parameter is not empty. You can get group ID's from <strong>/get/groups</strong> (Your contact groups).
                 * @apiParam {String} message Message you want to send, spintax and shortcodes are supported.
                 * @apiParam {Number} [shortener=none] Shortener ID, specify the shortener you want to use if you want to shorten the links in your message. You can get the list of available shorteners from <strong>/get/shorteners</strong>
                 *
                 * @apiExample {php} PHP Example
                  <?php

                    $chat = [
                        "secret" => "API_SECRET", // your API secret from (Tools -> API Keys) page
                        "account" => 1,
                        "numbers" => "+639123456789,+639123456789,+639123456789",
                        "groups" => "1,2,3,4",
                        "phone" => "+639123456789",
                        "message" => "Hello World!"
                    ];

                    $cURL = curl_init("http://127.0.0.1/zender/api/send/whatsapp.bulk");
                    curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    curl_setopt($cURL, CURLOPT_POSTFIELDS, $chat);
                    $response = curl_exec($cURL);
                    curl_close($cURL);

                    $result = json_decode($response, true);

                    // do something with response
                    print_r($result);
                 * 
                 * @apiExample {python} Python Example
                    import requests
    
                    # your API secret from (Tools -> API Keys) page
                    apiSecret = "API_SECRET"

                    chat = {
                        "secret": apiSecret,
                        "account": 1,
                        "numbers": "+639123456789,+639123456789,+639123456789",
                        "groups": "1,2,3,4",
                        "phone": "+639123456789",
                        "message": "Hello World!"
                    }

                    r = requests.post(url = "http://127.0.0.1/zender/api/send/whatsapp.bulk", params = chat)
                      
                    # do something with response object
                    result = r.json()
                 *
                 * @apiSuccess (Success Response Format) {Number} status List of Codes
                 * <br> 200 = Success
                 * @apiSuccess (Success Response Format) {String} message Response message
                 * @apiSuccess (Success Response Format) {Array} data Array of data
                 *
                 * @apiSuccessExample {json} Success Response
                 {
                   "status": 200,
                   "message": "WhatsApp message has been queued for sending!",
                   "data": false
                 }
                 * 
                 * @apiError (Error Response Format) {Number} status List of Codes<br>
                 * 400 = Invalid parameters<br>
                 * 401 = Invalid API secret<br>
                 * 403 = Access denied<br>
                 * 404 = WhatsApp account doesn't exist<br>
                 * 500 = Something went wrong
                 * @apiError (Error Response Format) {String} message Response message
                 * @apiError (Error Response Format) {Array} data Array of data
                 * 
                 * @apiErrorExample {json} Error Response
                 {
                   "status": 400,
                   "message": "Invalid Parameters!",
                   "data": false
                 }
                 *
                 */

                if(!in_array("wa_send_bulk", explode(",", $api["permissions"])))
                    response(403, "This API key doesn't have permission to use this endpoint!");

                if(!isset($request["account"], $request["message"]))
                    response(400, "Invalid Parameters!");

                if(!isset($request["numbers"]) && !isset($request["groups"]))
                    response(400, "Invalid Parameters!");

                if(!$this->sanitize->isInt($request["account"]))
                    response(400, "Invalid Parameters!");

                if(!$this->file->exists("system/storage/temporary/firebase.json")):
                    response(500, "System configuration error!");
                endif;

                if(!$this->sanitize->length($request["message"], system_message_min))
                    response(400, "Message is too short!");

                if(system_message_max > 0):
                    if($this->sanitize->length($request["message"], system_message_max, 2))
                        response(400, "Message is too long!");
                endif;

                $subscription = set_subscription(
                    $this->system->checkSubscription($api["uid"]), 
                    $this->system->getSubscription(false, $api["uid"]), 
                    $this->system->getSubscription(false, false, true)
                );

                if(empty($subscription))
                    response(403, "You are not subscribed to any premium package!");

                if($this->system->checkWaAccount($api["uid"], $request["account"], "id") < 1)
                    response(404, "WhatsApp account doesn't exist!");

                $contactBook = [];

                if(isset($request["numbers"])):
                    $numbers = explode(",", trim($request["numbers"]));

                    if(is_array($numbers) && !empty($numbers) && !empty($numbers[0])):
                        foreach($numbers as $number):
                            $rejected = false;

                            try {
                                $phone = $this->phone->parse($number, $api["country"]);

                                $phone->format(Brick\PhoneNumber\PhoneNumberFormat::INTERNATIONAL);

                                if(!$phone->isValidNumber())
                                    $rejected = true;

                                if(!$phone->getNumberType(Brick\PhoneNumber\PhoneNumberType::MOBILE))
                                    $rejected = true;

                                $phoneNumber = $phone->format(Brick\PhoneNumber\PhoneNumberFormat::E164);
                            } catch(Brick\PhoneNumber\PhoneNumberParseException $e) {
                                $rejected = true;
                            }

                            if(!$rejected):
                                $contactBook[] = [
                                    "name" => $phoneNumber,
                                    "phone" => $phoneNumber,
                                    "group" => "Unknown"
                                ];
                            endif;
                        endforeach;
                    endif;
                endif;

                if(isset($request["groups"])):
                    $groups = explode(",", trim($request["groups"]));

                    if(is_array($groups) && !empty($groups) && !empty($groups[0])):
                        foreach($request["groups"] as $group):
                            if($this->system->checkGroup($api["uid"], $group) > 0):
                                $contacts = $this->system->getContactsByGroup($api["uid"], $group);
                                
                                if(!empty($contacts)):
                                    foreach($contacts as $contact):
                                        try {
                                            $phone = $this->phone->parse($contact["phone"], $api["country"]);
                                        } catch(Brick\PhoneNumber\PhoneNumberParseException $e) {
                                            // ignore
                                        }

                                        $contactBook[] = [
                                            "name" => $contact["name"],
                                            "phone" => $contact["phone"],
                                            "group" => $contact["group"]
                                        ];
                                    endforeach;
                                endif;
                            endif;
                        endforeach;
                    endif;
                endif;

                if(empty($contactBook))
                    response(400, "Invalid Parameters!");

                if(isset($request["shortener"])):
                    if(!$this->sanitize->isInt($request["shortener"]))
                        response(400, "Invalid Parameters!");

                    if($request["shortener"] > 0):
                        if(!$this->file->exists("system/shorteners/" . md5($request["shortener"]) . ".php"))
                            response(404, "Specified shortener doesn't exist!");

                        $messageLinks = (new VStelmakh\UrlHighlight\UrlHighlight)->getUrls($request["message"]);

                        if(!empty($messageLinks)):
                            require "system/shorteners/" . md5($request["shortener"]) . ".php";

                            foreach($messageLinks as $key => $value):
                                $shortLink = shortenUrl($value, $this);

                                if($shortLink):
                                    $request["message"] = str_replace("{$value}", "{$shortLink}", $request["message"]);
                                endif;
                            endforeach;
                        endif;
                    endif;
                endif;

                $account = $this->system->getWaAccount($api["uid"], $request["account"], "id");

                $this->echo->_cache = $this->cache;
                $this->echo->_guzzle = $this->guzzle;

                try {
                    $echoToken = $this->echo->token();
                } catch(Exception $e){
                    response(500, "System configuration error!");
                }

                foreach($contactBook as $contact):
                    if(!limitation($subscription["wa_send_limit"], $this->system->countQuota($api["uid"], "wa_sent"))):
                        $this->system->create("wa_sent", [
                            "uid" => $api["uid"],
                            "wid" => $account["wid"],
                            "phone" => $contact["phone"],
                            "message" => bjoernffm\Spintax\Parser::parse($this->lex->parse(footermark($subscription["footermark"], $request["message"], system_message_mark), [
                                "contact" => [
                                    "name" => $contact["name"],
                                    "number" => $contact["phone"]
                                ],
                                "group" => [
                                    "name" => $contact["group"]
                                ],
                                "date" => [
                                    "now" => date("F j, Y"),
                                    "time" => date("h:i A") 
                                ]
                            ]))->generate(),
                            "status" => 1,
                            "api" => 1,
                            "create_date" => date("Y-m-d H:i:s", time())
                        ]);
                    endif;
                endforeach;

                if($echoToken):
                    $this->wa->_guzzle = $this->guzzle;

                    if($this->wa->send($echoToken, $account["unique"], $api["uid"], $api["hash"])):
                        $this->echo->notify($api["hash"], [
                            "type" => "table"
                        ]);

                        response(200, "WhatsApp bulk chats has been queued!");
                    else:
                        response(500, "Unable to connect to WhatsApp servers!");
                    endif;
                else:
                    response(500, "Something went wrong!");
                endif;

                break;
            case "ussd":
                /**
                 * @api {post} /send/ussd Send USSD Request
                 * @apiName Send USSD Request
                 * @apiDescription Send USSD Request. Requires "<strong>ussd</strong>" API permission.
                 * @apiGroup USSD
                 * @apiVersion 1.0.0
                 *
                 * @apiParam {String} secret The API secret you copied from (Tools -> API Keys) page
                 * @apiParam {String} code MMI request code. Please make sure that you are using a valid MMI code, if not, it will fail.
                 * @apiParam {Number} sim Sim slot number you want to use.
                 * @apiParam {String} device Linked device unique ID. You can get linked device unique ID from <strong>/get/devices</strong> (Your devices).
                 *
                 * @apiExample {php} PHP Example
                  <?php

                    $ussd = [
                        "secret" => "API_SECRET", // your API secret from (Tools -> API Keys) page
                        "code" => "*121#",
                        "sim" => 1,
                        "device" => "00000000-0000-0000-d57d-f30cb6a89289"
                    ];

                    $cURL = curl_init("http://127.0.0.1/zender/api/send/ussd");
                    curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    curl_setopt($cURL, CURLOPT_POSTFIELDS, $ussd);
                    $response = curl_exec($cURL);
                    curl_close($cURL);

                    $result = json_decode($response, true);

                    // do something with response
                    print_r($result);
                 * 
                 * @apiExample {python} Python Example
                    import requests
    
                    # your API secret from (Tools -> API Keys) page
                    apiSecret = "API_SECRET"

                    ussd = {
                        "secret": apiSecret,
                        "code": "*121#",
                        "sim": 1,
                        "device": "00000000-0000-0000-d57d-f30cb6a89289"
                    }

                    r = requests.post(url = "http://127.0.0.1/zender/api/send/ussd", params = ussd)
                      
                    # do something with response object
                    result = r.json()
                 *
                 * @apiSuccess (Success Response Format) {Number} status List of Codes
                 * <br> 200 = Success
                 * @apiSuccess (Success Response Format) {String} message Response message
                 * @apiSuccess (Success Response Format) {Array} data Array of data
                 *
                 * @apiSuccessExample {json} Success Response
                 {
                   "status": 200,
                   "message": "WhatsApp message has been queued for sending!",
                   "data": false
                 }
                 * 
                 * @apiError (Error Response Format) {Number} status List of Codes<br>
                 * 400 = Invalid parameters<br>
                 * 401 = Invalid API secret<br>
                 * 403 = Access denied<br> 
                 * 404 = Device doesn't exist<br>
                 * 500 = Something went wrong
                 * @apiError (Error Response Format) {String} message Response message
                 * @apiError (Error Response Format) {Array} data Array of data
                 * 
                 * @apiErrorExample {json} Error Response
                 {
                   "status": 400,
                   "message": "Invalid Parameters!",
                   "data": false
                 }
                 *
                 */

                if(!in_array("ussd", explode(",", $api["permissions"])))
                    response(403, "This API key doesn't have permission to use this endpoint!");

                if(!isset($request["code"], $request["sim"], $request["device"]))
                    response(400, "Invalid Parameters!");

                if(!$this->sanitize->isInt($request["sim"]))
                    response(400, "Invalid Parameters!");

                $subscription = set_subscription(
                    $this->system->checkSubscription($api["uid"]), 
                    $this->system->getSubscription(false, $api["uid"]), 
                    $this->system->getSubscription(false, false, true)
                );

                if(empty($subscription))
                    response(403, "You are not subscribed to any premium package!");

                if(limitation($subscription["ussd_limit"], $this->system->countQuota($api["uid"], "ussd")))
                    response(403, "Maximum number of USSD requests has been reached!");

                if($this->system->checkDevice($api["uid"], $request["device"], "did") < 1)
                    response(404, "Device doesn't exist!");

                $device = $this->system->getDevice($api["uid"], $request["device"], "did");

                if(!$device)
                    response(404, "Device doesn't exist!");

                $filtered = [
                    "uid" => $api["uid"],
                    "code" => $request["code"],
                    "did" => $request["device"],
                    "sim" => $request["sim"] < 2 ? 1 : 2,
                    "response" => false,
                    "status" => 1,
                    "create_date" => date("Y-m-d H:i:s", time())
                ];

                if($this->system->create("ussd", $filtered)):
                    if(round((int) $device["version"]) > 7):
                        $this->fcm->send(md5($api["uid"] . $request["device"]), [
                            "type" => "ussd"
                        ]);
                    else:
                        try {
                            $this->echo->_cache = $this->cache;
                            $this->echo->_guzzle = $this->guzzle;

                            $echoToken = $this->echo->token();

                            if($echoToken):
                                $this->echo->notify("legacy_" . md5($api["uid"] . $request["device"]), [
                                    "type" => "ussd"
                                ]);
                            endif;
                        } catch(Exception $e){
                            // Ignore
                        }
                    endif;

                    response(200, "USSD request has been queued!");
                else:
                    response(500, "Something went wrong!");
                endif;

                break;
            default:
                response(500, "Invalid API Endpoint!");
        endswitch;
    }

    public function clear()
    {
        $this->header->allow();

        $this->cache->container("system.settings");

        if($this->cache->empty()):
            $this->cache->setArray($this->system->getSettings());
        endif;

        set_system($this->cache->getAll());

        $request = $this->sanitize->array($_REQUEST);
        $type = $this->sanitize->string($this->url->segment(4));

        if(!isset($request["secret"]))
            response(400, "Invalid Parameters!");

        if($this->api->checkApikey($request["secret"]) < 1)
            response(401, "Invalid API secret supplied!");

        $api = $this->api->getApikey($request["secret"]);

        $permissions = explode(",", $api["permissions"]);

        if(!is_array($permissions))
            response(403, "Insufficient Permissions!");

        switch($type):
            case "sms":
                /**
                 * @api {get} /clear/sms Clear Pending SMS
                 * @apiName Clear Pending SMS
                 * @apiDescription Clear Pending SMS. Requires "<strong>clear_sms_pending</strong>" API permission.
                 * @apiGroup SMS
                 * @apiVersion 1.0.0
                 *
                 * @apiParam {String} secret The API secret you copied from (Tools -> API Keys) page
                 * 
                 * @apiExample {php} PHP Example
                  <?php

                    $apiSecret = "API_SECRET"; // your API secret from (Tools -> API Keys) page

                    $cURL = curl_init();
                    curl_setopt($cURL, CURLOPT_URL, "http://127.0.0.1/zender/api/clear/sms?secret={$apiSecret}");
                    curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    $response = curl_exec($cURL);
                    curl_close($cURL);

                    $result = json_decode($response, true);

                    // do something with response
                    print_r($result);
                 * 
                 * @apiExample {python} Python Example
                    import requests
    
                    # your API secret from (Tools -> API Keys) page
                    apiSecret = "API_SECRET"

                    r = requests.get(url = "http://127.0.0.1/zender/api/clear/sms", params = {
                        "secret": apiSecret
                    })
                      
                    # do something with response object
                    result = r.json()
                 * 
                 * @apiSuccess (Success Response Format) {Number} status List of Codes
                 * <br> 200 = Success
                 * @apiSuccess (Success Response Format) {String} message Response message
                 * @apiSuccess (Success Response Format) {Array} data Array of data
                 *
                 * @apiSuccessExample {json} Success Response
                 {
                   "status": 200,
                   "message": "Pending sms messages has been cleared!",
                   "data": false
                 }
                 * 
                 * @apiError (Error Response Format) {Number} status List of Codes<br>
                 * 400 = Invalid parameters<br>
                 * 401 = Invalid API secret<br>
                 * 403 = Access denied<br> 
                 * 404 = Device doesn't exist<br>
                 * 500 = Something went wrong
                 * @apiError (Error Response Format) {String} message Response message
                 * @apiError (Error Response Format) {Array} data Array of data
                 * 
                 * @apiErrorExample {json} Error Response
                 {
                   "status": 400,
                   "message": "Invalid Parameters!",
                   "data": false
                 }
                 *
                 */

                if(!in_array("clear_sms_pending", explode(",", $api["permissions"])))
                    response(403, "This API key doesn't have permission to use this endpoint!");

                if($this->system->clearPending($api["uid"], 1)):
                    $this->fcm->send($this->hash->encode($api["uid"], system_token), [
                        "type" => "clear",
                        "data" => "sms",
                        "uid" => (int) $api["uid"]
                    ]);

                    try {
                        $this->echo->_cache = $this->cache;
                        $this->echo->_guzzle = $this->guzzle;

                        $echoToken = $this->echo->token();

                        if($echoToken):
                            $this->echo->notify("legacy_{$this->hash->encode($api["uid"], system_token)}", [
                                "type" => "clear",
                                "data" => "sms",
                                "uid" => (int) $api["uid"]
                            ]);
                        endif;
                    } catch(Exception $e){
                        // Ignore
                    }
                endif;

                response(200, "Pending sms messages has been cleared!");

                break;
            case "ussd":
                /**
                 * @api {get} /clear/ussd Clear Pending USSD
                 * @apiName Clear Pending USSD
                 * @apiDescription Clear Pending USSD. Requires "<strong>clear_ussd_pending</strong>" API permission.
                 * @apiGroup USSD
                 * @apiVersion 1.0.0
                 *
                 * @apiParam {String} secret The API secret you copied from (Tools -> API Keys) page
                 *
                 * @apiExample {php} PHP Example 
                  <?php

                    $apiSecret = "API_SECRET"; // your API secret from (Tools -> API Keys) page

                    $cURL = curl_init();
                    curl_setopt($cURL, CURLOPT_URL, "http://127.0.0.1/zender/api/clear/ussd?secret={$apiSecret}");
                    curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    $response = curl_exec($cURL);
                    curl_close($cURL);

                    $result = json_decode($response, true);

                    // do something with response
                    print_r($result);
                 * 
                 * @apiExample {python} Python Example
                    import requests
    
                    # your API secret from (Tools -> API Keys) page
                    apiSecret = "API_SECRET"

                    r = requests.get(url = "http://127.0.0.1/zender/api/clear/ussd", params = {
                        "secret": apiSecret
                    })
                      
                    # do something with response object
                    result = r.json()
                 * 
                 * @apiSuccess (Success Response Format) {Number} status List of Codes
                 * <br> 200 = Success
                 * @apiSuccess (Success Response Format) {String} message Response message
                 * @apiSuccess (Success Response Format) {Array} data Array of data
                 *
                 * @apiSuccessExample {json} Success Response
                 {
                   "status": 200,
                   "message": "Pending USSD requests has been cleared!",
                   "data": false
                 }
                 * 
                 * @apiError (Error Response Format) {Number} status List of Codes<br>
                 * 400 = Invalid parameters<br>
                 * 401 = Invalid API secret<br>
                 * 403 = Access denied<br> 
                 * 404 = Device doesn't exist<br>
                 * 500 = Something went wrong
                 * @apiError (Error Response Format) {String} message Response message
                 * @apiError (Error Response Format) {Array} data Array of data
                 * 
                 * @apiErrorExample {json} Error Response
                 {
                   "status": 400,
                   "message": "Invalid Parameters!",
                   "data": false
                 }
                 *
                 */

                if(!in_array("clear_ussd_pending", explode(",", $api["permissions"])))
                    response(403, "This API key doesn't have permission to use this endpoint!");

                if($this->system->clearPending($api["uid"], 2)):
                    $this->fcm->send($this->hash->encode($api["uid"], system_token), [
                        "type" => "clear",
                        "data" => "ussd",
                        "uid" => (int) $api["uid"]
                    ]);

                    try {
                        $this->echo->_cache = $this->cache;
                        $this->echo->_guzzle = $this->guzzle;

                        $echoToken = $this->echo->token();

                        if($echoToken):
                            $this->echo->notify("legacy_{$this->hash->encode($api["uid"], system_token)}", [
                                "type" => "clear",
                                "data" => "ussd",
                                "uid" => (int) $api["uid"]
                            ]);
                        endif;
                    } catch(Exception $e){
                        // Ignore
                    }
                endif;

                response(200, "Pending USSD requests has been cleared!");

                break;
            default:
                response(500, "Invalid API Endpoint!");
        endswitch;
    }

	public function get()
	{
		$this->header->allow();

        $this->cache->container("system.settings");

        if($this->cache->empty()):
            $this->cache->setArray($this->system->getSettings());
        endif;

        set_system($this->cache->getAll());

        $request = $this->sanitize->array($_REQUEST);
        $type = $this->sanitize->string($this->url->segment(4));

        if(!isset($request["secret"]))
            response(400, "Invalid Parameters!");

        if($this->api->checkApikey($request["secret"]) < 1)
            response(401, "Invalid API secret supplied!");

        $api = $this->api->getApikey($request["secret"]);

        $permissions = explode(",", $api["permissions"]);

        if(!is_array($permissions))
            response(403, "Insufficient Permissions!");

        switch($type):
            case "otp":
                /**
                 * @api {get} /get/otp Verify OTP
                 * @apiName Verify OTP
                 * @apiDescription Verify one-time-password from user supplied data. Requires "<strong>otp</strong>" API permission.
                 * @apiGroup OTP
                 * @apiVersion 1.0.0
                 *
                 * @apiParam {String} secret The API secret you copied from (Tools -> API Keys) page
                 * @apiParam {String} otp The otp you got from a user supplied input or data
                 *
                 * @apiExample {php} PHP Example
                  <?php

                    $apiSecret = "API_SECRET"; // your API secret from (Tools -> API Keys) page
                    $otp = "123456"; // otp from a user supplied input or data

                    $cURL = curl_init();
                    curl_setopt($cURL, CURLOPT_URL, "http://127.0.0.1/zender/api/get/credits?secret={$apiSecret}&otp={$otp}");
                    curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    $response = curl_exec($cURL);
                    curl_close($cURL);

                    $result = json_decode($response, true);

                    // do something with response
                    print_r($result);
                 * 
                 * @apiExample {python} Python Example
                    import requests
    
                    # your API secret from (Tools -> API Keys) page
                    apiSecret = "API_SECRET"

                    # otp from a user supplied input or data
                    otpCode = "123456"

                    r = requests.get(url = "http://127.0.0.1/zender/api/get/credits", params = {
                        "secret": apiSecret,
                        "otp": otpCode
                    })
                      
                    # do something with response object
                    result = r.json()
                 * 
                 * @apiSuccess (Success Response Format) {Number} status List of Codes
                 * <br> 200 = Success
                 * @apiSuccess (Success Response Format) {String} message Response message
                 * @apiSuccess (Success Response Format) {Array} data Array of data
                 *
                 * @apiSuccessExample {json} Success Response
                 {
                   "status": 200,
                   "message": "OTP has been verified!",
                   "data": false
                 }
                 * 
                 * @apiError (Error Response Format) {Number} status List of Codes<br>
                 * 400 = Invalid parameters<br>
                 * 401 = Invalid API secret<br>
                 * 403 = Access denied<br>
                 * 500 = Something went wrong
                 * @apiError (Error Response Format) {String} message Response message
                 * @apiError (Error Response Format) {Array} data Array of data
                 * 
                 * @apiErrorExample {json} Error Response
                 {
                   "status": 403,
                   "message": "OTP is invalid or expired!",
                   "data": false
                 }
                 *
                 */

                if(!in_array("otp", explode(",", $api["permissions"])))
                    response(403, "This API key doesn't have permission to use this endpoint!");

                if(!isset($request["otp"]))
                    response(400, "Invalid Parameters!");

                $this->cache->container("api.otp.{$request["secret"]}", true);

                if($this->cache->has($request["otp"])):
                    $this->cache->delete($request["otp"]);
                    response(200, "OTP has been verified!");
                endif;

                response(403, "OTP is invalid or expired!");

                break;
            case "credits":
                /**
                 * @api {get} /get/credits Get Remaining Credits
                 * @apiName Get Remaining Credits
                 * @apiDescription Get Remaining Credits. Requires "<strong>get_credits</strong>" API permission.
                 * @apiGroup Account
                 * @apiVersion 1.0.0
                 *
                 * @apiParam {String} secret The API secret you copied from (Tools -> API Keys) page
                 *
                 * @apiExample {php} PHP Example
                  <?php

                    $apiSecret = "API_SECRET"; // your API secret from (Tools -> API Keys) page

                    $cURL = curl_init();
                    curl_setopt($cURL, CURLOPT_URL, "http://127.0.0.1/zender/api/get/credits?secret={$apiSecret}");
                    curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    $response = curl_exec($cURL);
                    curl_close($cURL);

                    $result = json_decode($response, true);

                    // do something with response
                    print_r($result);
                 * 
                 * @apiExample {python} Python Example
                    import requests
    
                    # your API secret from (Tools -> API Keys) page
                    apiSecret = "API_SECRET"

                    r = requests.get(url = "http://127.0.0.1/zender/api/get/credits", params = {
                        "secret": apiSecret
                    })
                      
                    # do something with response object
                    result = r.json()
                 * 
                 * @apiSuccess (Success Response Format) {Number} status List of Codes
                 * <br> 200 = Success
                 * @apiSuccess (Success Response Format) {String} message Response message
                 * @apiSuccess (Success Response Format) {Array} data Array of data
                 *
                 * @apiSuccessExample {json} Success Response
                 {
                   "status": 200,
                   "message": "Remaining Credits",
                   "data": {
                        "credits": "798.634",
                        "currency": "GBP"
                    }
                 }
                 * 
                 * @apiError (Error Response Format) {Number} status List of Codes<br>
                 * 400 = Invalid parameters<br>
                 * 401 = Invalid API secret<br>
                 * 403 = Access denied<br>
                 * 500 = Something went wrong
                 * @apiError (Error Response Format) {String} message Response message
                 * @apiError (Error Response Format) {Array} data Array of data
                 * 
                 * @apiErrorExample {json} Error Response
                 {
                   "status": 400,
                   "message": "Invalid Parameters!",
                   "data": false
                 }
                 *
                 */

                if(!in_array("get_credits", explode(",", $api["permissions"])))
                    response(403, "This API key doesn't have permission to use this endpoint!");

                response(200, "Remaining Credits", [
                    "credits" => $api["credits"],
                    "currency" => system_currency
                ]);

                break;
            case "earnings":
                /**
                 * @api {get} /get/earnings Get Partner Earnings
                 * @apiName Get Partner Earnings
                 * @apiDescription Get Partner Earnings. Requires "<strong>get_earnings</strong>" API permission.
                 * @apiGroup Account
                 * @apiVersion 1.0.0
                 *
                 * @apiParam {String} secret The API secret you copied from (Tools -> API Keys) page
                 *
                 * @apiExample {php} PHP Example
                  <?php

                    $apiSecret = "API_SECRET"; // your API secret from (Tools -> API Keys) page

                    $cURL = curl_init();
                    curl_setopt($cURL, CURLOPT_URL, "http://127.0.0.1/zender/api/get/earnings?secret={$apiSecret}");
                    curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    $response = curl_exec($cURL);
                    curl_close($cURL);

                    $result = json_decode($response, true);

                    // do something with response
                    print_r($result);
                 * 
                 * @apiExample {python} Python Example
                    import requests
    
                    # your API secret from (Tools -> API Keys) page
                    apiSecret = "API_SECRET"

                    r = requests.get(url = "http://127.0.0.1/zender/api/get/earnings", params = {
                        "secret": apiSecret
                    })
                      
                    # do something with response object
                    result = r.json()
                 * 
                 * @apiSuccess (Success Response Format) {Number} status List of Codes
                 * <br> 200 = Success
                 * @apiSuccess (Success Response Format) {String} message Response message
                 * @apiSuccess (Success Response Format) {Array} data Array of data
                 *
                 * @apiSuccessExample {json} Success Response
                 {
                   "status": 200,
                   "message": "Partner Credits",
                   "data": {
                        "earnings": "1.43638",
                        "currency": "GBP"
                    }
                 }
                 * 
                 * @apiError (Error Response Format) {Number} status List of Codes<br>
                 * 400 = Invalid parameters<br>
                 * 401 = Invalid API secret<br>
                 * 403 = Access denied<br>
                 * 500 = Something went wrong
                 * @apiError (Error Response Format) {String} message Response message
                 * @apiError (Error Response Format) {Array} data Array of data
                 * 
                 * @apiErrorExample {json} Error Response
                 {
                   "status": 400,
                   "message": "Invalid Parameters!",
                   "data": false
                 }
                 *
                 */

                if(!in_array("get_earnings", explode(",", $api["permissions"])))
                    response(403, "This API key doesn't have permission to use this endpoint!");

                response(200, "Partner Earnings", [
                    "earnings" => $api["earnings"],
                    "currency" => system_currency
                ]);

                break;
            case "subscription":
                /**
                 * @api {get} /get/subscription Get Subscription Package
                 * @apiName Get Subscription Package
                 * @apiDescription Get Subscription Package. Requires "<strong>get_subscription</strong>" API permission.
                 * @apiGroup Account
                 * @apiVersion 1.0.0
                 *
                 * @apiParam {String} secret The API secret you copied from (Tools -> API Keys) page
                 *
                 * @apiExample {php} PHP Example
                  <?php

                    $apiSecret = "API_SECRET"; // your API secret from (Tools -> API Keys) page

                    $cURL = curl_init();
                    curl_setopt($cURL, CURLOPT_URL, "http://127.0.0.1/zender/api/get/subscription?secret={$apiSecret}");
                    curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    $response = curl_exec($cURL);
                    curl_close($cURL);

                    $result = json_decode($response, true);

                    // do something with response
                    print_r($result);
                 * 
                 * @apiExample {python} Python Example
                    import requests
    
                    # your API secret from (Tools -> API Keys) page
                    apiSecret = "API_SECRET"

                    r = requests.get(url = "http://127.0.0.1/zender/api/get/subscription", params = {
                        "secret": apiSecret
                    })
                      
                    # do something with response object
                    result = r.json()
                 * 
                 * @apiSuccess (Success Response Format) {Number} status List of Codes
                 * <br> 200 = Success
                 * @apiSuccess (Success Response Format) {String} message Response message
                 * @apiSuccess (Success Response Format) {Array} data Array of data
                 *
                 * @apiSuccessExample {json} Success Response
                 {
                   "status": 200,
                   "message": "Subscription Package",
                   "data": "data": {
                    "name": "Starter",
                    "usage": {
                            "sms_send": {
                                "used": 262,
                                "limit": 1000
                            },
                            "sms_receive": {
                                "used": 139,
                                "limit": 250
                            },
                            "ussd": {
                                "used": 16,
                                "limit": 0
                            },
                            "notifications": {
                                "used": 55,
                                "limit": 0
                            },
                            "contacts": {
                                "used": 7,
                                "limit": 50
                            },
                            "devices": {
                                "used": 3,
                                "limit": 3
                            },
                            "apikeys": {
                                "used": 4,
                                "limit": 5
                            },
                            "webhooks": {
                                "used": 1,
                                "limit": 5
                            },
                            "actions": {
                                "used": 3,
                                "limit": 0
                            },
                            "scheduled": {
                                "used": 0,
                                "limit": 0
                            },
                            "wa_send": {
                                "used": 3,
                                "limit": 0
                            },
                            "wa_receive": {
                                "used": 19,
                                "limit": 0
                            },
                            "wa_accounts": {
                                "used": 1,
                                "limit": 0
                            }
                        }
                    }
                 }
                 * 
                 * @apiError (Error Response Format) {Number} status List of Codes<br>
                 * 400 = Invalid parameters<br>
                 * 401 = Invalid API secret<br>
                 * 403 = Access denied<br>
                 * 500 = Something went wrong
                 * @apiError (Error Response Format) {String} message Response message
                 * @apiError (Error Response Format) {Array} data Array of data
                 * 
                 * @apiErrorExample {json} Error Response
                 {
                   "status": 400,
                   "message": "Invalid Parameters!",
                   "data": false
                 }
                 *
                 */

                if(!in_array("get_subscription", explode(",", $api["permissions"])))
                    response(403, "This API key doesn't have permission to use this endpoint!");

                $subscription = set_subscription(
                    $this->system->checkSubscription($api["uid"]), 
                    $this->system->getSubscription(false, $api["uid"]), 
                    $this->system->getSubscription(false, false, true)
                );

                if(empty($subscription))
                    response(403, "Account is not subscribed to any premium package!");

                $package = [
                    "name" => $subscription["name"],
                    "usage" => [
                        "sms_send" => [
                            "used" => (int) $this->system->countQuota($api["uid"], "sent"),
                            "limit" => (int) $subscription["send_limit"]
                        ],
                        "sms_receive" => [
                            "used" => (int) $this->system->countQuota($api["uid"], "received"),
                            "limit" => (int) $subscription["receive_limit"]
                        ],
                        "ussd" => [
                            "used" => (int) $this->system->countQuota($api["uid"], "ussd"),
                            "limit" => (int) $subscription["ussd_limit"]
                        ],
                        "notifications" => [
                            "used" => (int) $this->system->countQuota($api["uid"], "notifications"),
                            "limit" => (int) $subscription["notification_limit"]
                        ],
                        "contacts" => [
                            "used" => (int) $this->system->countContacts($api["uid"]),
                            "limit" => (int) $subscription["contact_limit"]
                        ],
                        "devices" => [
                            "used" => (int) $this->system->countDevices($api["uid"]),
                            "limit" => (int) $subscription["device_limit"]
                        ],
                        "apikeys" => [
                            "used" => (int) $this->system->countKeys($api["uid"]),
                            "limit" => (int) $subscription["key_limit"]
                        ],
                        "webhooks" => [
                            "used" => (int) $this->system->countWebhooks($api["uid"]),
                            "limit" => (int) $subscription["webhook_limit"]
                        ],
                        "actions" => [
                            "used" => (int) $this->system->countActions($api["uid"]),
                            "limit" => (int) $subscription["action_limit"]
                        ],
                        "scheduled" => [
                            "used" => (int) $this->system->countScheduled($api["uid"]),
                            "limit" => (int) $subscription["scheduled_limit"]
                        ],
                        "wa_send" => [
                            "used" => (int) $this->system->countQuota($api["uid"], "wa_sent"),
                            "limit" => (int) $subscription["wa_send_limit"]
                        ],
                        "wa_receive" => [
                            "used" => (int) $this->system->countQuota($api["uid"], "wa_received"),
                            "limit" => (int) $subscription["wa_receive_limit"]
                        ],
                        "wa_accounts" => [
                            "used" => (int) $this->system->countWaAccounts($api["uid"]),
                            "limit" => (int) $subscription["wa_account_limit"]
                        ]
                    ]
                ];

                response(200, "Subscription Package", $package);

                break;
            case "sms.sent":
                /**
                 * @api {get} /get/sms.sent Get Sent Messages
                 * @apiName Get Sent Messages
                 * @apiDescription Get Sent Messages. Requires "<strong>get_sms_sent</strong>" API permission.
                 * @apiGroup SMS
                 * @apiVersion 1.0.0
                 *
                 * @apiParam {String} secret The API secret you copied from (Tools -> API Keys) page
                 * @apiParam {Number} [limit=10] Limit the number of results per page.
                 * @apiParam {Number} [page=1] Pagination of results.
                 *
                 * @apiExample {php} PHP Example
                  <?php

                    $apiSecret = "API_SECRET"; // your API secret from (Tools -> API Keys) page

                    $cURL = curl_init();
                    curl_setopt($cURL, CURLOPT_URL, "http://127.0.0.1/zender/api/get/sms.sent?secret={$apiSecret}");
                    curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    $response = curl_exec($cURL);
                    curl_close($cURL);

                    $result = json_decode($response, true);

                    // do something with response
                    print_r($result);
                 * 
                 * @apiExample {python} Python Example
                    import requests
    
                    # your API secret from (Tools -> API Keys) page
                    apiSecret = "API_SECRET"

                    r = requests.get(url = "http://127.0.0.1/zender/api/get/sms.sent", params = {
                        "secret": apiSecret
                    })
                      
                    # do something with response object
                    result = r.json()
                 * 
                 * @apiSuccess (Success Response Format) {Number} status List of Codes
                 * <br> 200 = Success
                 * @apiSuccess (Success Response Format) {String} message Response message
                 * @apiSuccess (Success Response Format) {Array} data Array of data
                 *
                 * @apiSuccessExample {json} Success Response
                 {
                   "status": 200,
                   "message": "Sent SMS Messages",
                   "data": [
                        {
                            "mode": "Devices",
                            "sender": "00000000-0000-0000-d57d-f30cb6a89289",
                            "sender_type": "device",
                            "sim": 2,
                            "priority": false,
                            "api": false,
                            "status": "sent",
                            "status_code": "SMS_SENT",
                            "recipient": "+639206150513",
                            "message": "Hello World!",
                            "created": 1644382599
                        },
                        {
                            "mode": "Devices",
                            "sender": "00000000-0000-0000-d57d-f30cb6a89289",
                            "sender_type": "device",
                            "sim": 2,
                            "priority": false,
                            "api": false,
                            "status": "sent",
                            "status_code": "SMS_SENT",
                            "recipient": "+639184661533",
                            "message": "Hello World!",
                            "created": 1644382597
                        },
                        {
                            "mode": "Credits",
                            "sender": "Twilio",
                            "sender_type": "gateway",
                            "sim": 0,
                            "priority": false,
                            "api": false,
                            "status": "sent",
                            "status_code": "",
                            "recipient": "+639206150513",
                            "message": "Hello World!",
                            "created": 1644382807
                        }
                    ]
                 }
                 * 
                 * @apiError (Error Response Format) {Number} status List of Codes<br>
                 * 400 = Invalid parameters<br>
                 * 401 = Invalid API secret<br>
                 * 403 = Access denied<br>
                 * 500 = Something went wrong
                 * @apiError (Error Response Format) {String} message Response message
                 * @apiError (Error Response Format) {Array} data Array of data
                 * 
                 * @apiErrorExample {json} Error Response
                 {
                   "status": 400,
                   "message": "Invalid Parameters!",
                   "data": false
                 }
                 *
                 */

                if(!in_array("get_sms_sent", explode(",", $api["permissions"])))
                    response(403, "This API key doesn't have permission to use this endpoint!");

                if(isset($request["page"])):
                    if(!$this->sanitize->isInt($request["page"]))
                        response(400, "Invalid Parameters!");

                    $page = $request["page"];
                else:
                    $page = 1;
                endif;

                if(isset($request["limit"])):
                    if(!$this->sanitize->isInt($request["limit"]))
                        response(400, "Invalid Parameters!");

                    $limit = $request["limit"];
                else:
                    $limit = 10;
                endif;

                $messages = $this->api->getSms($api["uid"], "sent", "sent", abs($page), abs($limit));

                $messageArray = [];

                if(!empty($messages)):
                    foreach($messages as $message):
                        $messageArray[] = [
                            "mode" => $message["mode"] < 2 ? "Devices" : "Credits",
                            "sender" => $message["mode"] < 2 ? $message["did"] : ($message["gateway"] > 0 ? $message["gateway_name"] : $message["did"]),
                            "sender_type" => $message["mode"] < 2 ? "device" : ($message["gateway"] > 0 ? "gateway" : "partner_device"),
                            "sim" => (int) ($message["mode"] < 2 ? $message["sim"] : ($message["gateway"] > 0 ? 0 : $message["sim"])),
                            "priority" => $message["mode"] < 2 ? ($message["priority"] < 2 ? false : true) : ($message["gateway"] > 0 ? false : ($message["priority"] < 2 ? false : true)),
                            "api" => $message["api"] < 2 ? true : false,
                            "status" => $message["status"] < 4 ? "sent" : "failed",
                            "status_code" => $message["status_code"],
                            "recipient" => $message["phone"],
                            "message" => $message["message"],
                            "created" => strtotime($message["create_date"])
                        ];
                    endforeach;
                endif;

                response(200, "Sent SMS Messages", $messageArray);

                break;
            case "sms.pending":
                /**
                 * @api {get} /get/sms.pending Get Pending Messages
                 * @apiName Get Pending Messages
                 * @apiDescription Get Pending Messages. Requires "<strong>get_sms_pending</strong>" API permission.
                 * @apiGroup SMS
                 * @apiVersion 1.0.0
                 *
                 * @apiParam {String} secret The API secret you copied from (Tools -> API Keys) page
                 * @apiParam {Number} [limit=10] Limit the number of results per page.
                 * @apiParam {Number} [page=1] Pagination of results.
                 *
                 * @apiExample {php} PHP Example
                  <?php

                    $apiSecret = "API_SECRET"; // your API secret from (Tools -> API Keys) page

                    $cURL = curl_init();
                    curl_setopt($cURL, CURLOPT_URL, "http://127.0.0.1/zender/api/get/sms.pending?secret={$apiSecret}");
                    curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    $response = curl_exec($cURL);
                    curl_close($cURL);

                    $result = json_decode($response, true);

                    // do something with response
                    print_r($result);
                 * 
                 * @apiExample {python} Python Example
                    import requests
    
                    # your API secret from (Tools -> API Keys) page
                    apiSecret = "API_SECRET"

                    r = requests.get(url = "http://127.0.0.1/zender/api/get/sms.pending", params = {
                        "secret": apiSecret
                    })
                      
                    # do something with response object
                    result = r.json()
                 * 
                 * @apiSuccess (Success Response Format) {Number} status List of Codes
                 * <br> 200 = Success
                 * @apiSuccess (Success Response Format) {String} message Response message
                 * @apiSuccess (Success Response Format) {Array} data Array of data
                 *
                 * @apiSuccessExample {json} Success Response
                 {
                   "status": 200,
                   "message": "Pending SMS Messages",
                   "data": [
                        {
                            "mode": "Devices",
                            "sender": "00000000-0000-0000-d57d-f30cb6a89289",
                            "sender_type": "device",
                            "sim": 2,
                            "priority": false,
                            "api": false,
                            "recipient": "+639184661533",
                            "message": "Hello World!",
                            "created": 1645520349
                        },
                        {
                            "mode": "Devices",
                            "sender": "00000000-0000-0000-d57d-f30cb6a89289",
                            "sender_type": "device",
                            "sim": 2,
                            "priority": false,
                            "api": false,
                            "recipient": "+639206150513",
                            "message": "Hello World!",
                            "created": 1645520349
                        },
                        {
                            "mode": "Credits",
                            "sender": "00000000-0000-0000-d57d-f30cb6a89289",
                            "sender_type": "partner_device",
                            "sim": 2,
                            "priority": false,
                            "api": false,
                            "recipient": "+639184661532",
                            "message": "Hello World!",
                            "created": 1645520349
                        }
                    ]
                 }
                 * 
                 * @apiError (Error Response Format) {Number} status List of Codes<br>
                 * 400 = Invalid parameters<br>
                 * 401 = Invalid API secret<br>
                 * 403 = Access denied<br>
                 * 500 = Something went wrong
                 * @apiError (Error Response Format) {String} message Response message
                 * @apiError (Error Response Format) {Array} data Array of data
                 * 
                 * @apiErrorExample {json} Error Response
                 {
                   "status": 400,
                   "message": "Invalid Parameters!",
                   "data": false
                 }
                 *
                 */

                if(!in_array("get_sms_pending", explode(",", $api["permissions"])))
                    response(403, "This API key doesn't have permission to use this endpoint!");

                if(isset($request["page"])):
                    if(!$this->sanitize->isInt($request["page"]))
                        response(400, "Invalid Parameters!");

                    $page = $request["page"];
                else:
                    $page = 1;
                endif;

                if(isset($request["limit"])):
                    if(!$this->sanitize->isInt($request["limit"]))
                        response(400, "Invalid Parameters!");

                    $limit = $request["limit"];
                else:
                    $limit = 10;
                endif;

                $messages = $this->api->getSms($api["uid"], "sent", "pending", abs($page), abs($limit));

                $messageArray = [];

                if(!empty($messages)):
                    foreach($messages as $message):
                        $messageArray[] = [
                            "mode" => $message["mode"] < 2 ? "Devices" : "Credits",
                            "sender" => $message["mode"] < 2 ? $message["did"] : ($message["gateway"] > 0 ? $message["gateway_name"] : $message["did"]),
                            "sender_type" => $message["mode"] < 2 ? "device" : ($message["gateway"] > 0 ? "gateway" : "partner_device"),
                            "sim" => (int) ($message["mode"] < 2 ? $message["sim"] : ($message["gateway"] > 0 ? 0 : $message["sim"])),
                            "priority" => $message["mode"] < 2 ? ($message["priority"] < 2 ? false : true) : ($message["gateway"] > 0 ? false : ($message["priority"] < 2 ? false : true)),
                            "api" => $message["api"] < 2 ? true : false,
                            "recipient" => $message["phone"],
                            "message" => $message["message"],
                            "created" => strtotime($message["create_date"])
                        ];
                    endforeach;
                endif;

                response(200, "Pending SMS Messages", $messageArray);

                break;
            case "sms.received":
                /**
                 * @api {get} /get/sms.received Get Received Messages
                 * @apiName Get Received Messages
                 * @apiDescription Get Received Messages. Requires "<strong>get_sms_received</strong>" API permission.
                 * @apiGroup SMS
                 * @apiVersion 1.0.0
                 *
                 * @apiParam {String} secret The API secret you copied from (Tools -> API Keys) page
                 * @apiParam {Number} [limit=10] Limit the number of results per page.
                 * @apiParam {Number} [page=1] Pagination of results.
                 *
                 * @apiExample {php} PHP Example
                  <?php

                    $apiSecret = "API_SECRET"; // your API secret from (Tools -> API Keys) page

                    $cURL = curl_init();
                    curl_setopt($cURL, CURLOPT_URL, "http://127.0.0.1/zender/api/get/sms.received?secret={$apiSecret}");
                    curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    $response = curl_exec($cURL);
                    curl_close($cURL);

                    $result = json_decode($response, true);

                    // do something with response
                    print_r($result);
                 * 
                 * @apiExample {python} Python Example
                    import requests
    
                    # your API secret from (Tools -> API Keys) page
                    apiSecret = "API_SECRET"

                    r = requests.get(url = "http://127.0.0.1/zender/api/get/sms.received", params = {
                        "secret": apiSecret
                    })
                      
                    # do something with response object
                    result = r.json()
                 * 
                 * @apiSuccess (Success Response Format) {Number} status List of Codes
                 * <br> 200 = Success
                 * @apiSuccess (Success Response Format) {String} message Response message
                 * @apiSuccess (Success Response Format) {Array} data Array of data
                 *
                 * @apiSuccessExample {json} Success Response
                 {
                   "status": 200,
                   "message": "Received SMS Messages",
                   "data": [
                        {
                            "device": "00000000-0000-0000-d57d-f30cb6a89289",
                            "sender": "+639760713666",
                            "message": "Hello World!",
                            "created": 1644405663
                        },
                        {
                            "device": "00000000-0000-0000-d57d-f30cb6a89289",
                            "sender": "GCash",
                            "message": "Hello World!",
                            "created": 1644417283
                        },
                        {
                            "device": "00000000-0000-0000-d57d-f30cb6a89289",
                            "sender": "TWILIO",
                            "message": "Hello World!",
                            "created": 1644421353
                        }
                    ]
                 }
                 * 
                 * @apiError (Error Response Format) {Number} status List of Codes<br>
                 * 400 = Invalid parameters<br>
                 * 401 = Invalid API secret<br>
                 * 403 = Access denied<br>
                 * 500 = Something went wrong
                 * @apiError (Error Response Format) {String} message Response message
                 * @apiError (Error Response Format) {Array} data Array of data
                 * 
                 * @apiErrorExample {json} Error Response
                 {
                   "status": 400,
                   "message": "Invalid Parameters!",
                   "data": false
                 }
                 *
                 */

                if(!in_array("get_sms_received", explode(",", $api["permissions"])))
                    response(403, "This API key doesn't have permission to use this endpoint!");

                if(isset($request["page"])):
                    if(!$this->sanitize->isInt($request["page"]))
                        response(400, "Invalid Parameters!");

                    $page = $request["page"];
                else:
                    $page = 1;
                endif;

                if(isset($request["limit"])):
                    if(!$this->sanitize->isInt($request["limit"]))
                        response(400, "Invalid Parameters!");

                    $limit = $request["limit"];
                else:
                    $limit = 10;
                endif;

                $messages = $this->api->getSms($api["uid"], "received", false, abs($page), abs($limit));

                $messageArray = [];

                if(!empty($messages)):
                    foreach($messages as $message):
                        $messageArray[] = [
                            "device" => $message["did"],
                            "sender" => $message["phone"],
                            "message" => $message["message"],
                            "created" => strtotime($message["receive_date"])
                        ];
                    endforeach;
                endif;

                response(200, "Received SMS Messages", $messageArray);

                break;
            case "wa.sent":
                /**
                 * @api {get} /get/wa.sent Get Sent Chats
                 * @apiName Get Sent Chats
                 * @apiDescription Get Sent Chats. Requires "<strong>get_wa_sent</strong>" API permission.
                 * @apiGroup WhatsApp
                 * @apiVersion 1.0.0
                 *
                 * @apiParam {String} secret The API secret you copied from (Tools -> API Keys) page
                 * @apiParam {Number} [limit=10] Limit the number of results per page.
                 * @apiParam {Number} [page=1] Pagination of results.
                 *
                 * @apiExample {php} PHP Example
                  <?php

                    $apiSecret = "API_SECRET"; // your API secret from (Tools -> API Keys) page

                    $cURL = curl_init();
                    curl_setopt($cURL, CURLOPT_URL, "http://127.0.0.1/zender/api/get/wa.sent?secret={$apiSecret}");
                    curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    $response = curl_exec($cURL);
                    curl_close($cURL);

                    $result = json_decode($response, true);

                    // do something with response
                    print_r($result);
                 * 
                 * @apiExample {python} Python Example
                    import requests
    
                    # your API secret from (Tools -> API Keys) page
                    apiSecret = "API_SECRET"

                    r = requests.get(url = "http://127.0.0.1/zender/api/get/wa.sent", params = {
                        "secret": apiSecret
                    })
                      
                    # do something with response object
                    result = r.json()
                 * 
                 * @apiSuccess (Success Response Format) {Number} status List of Codes
                 * <br> 200 = Success
                 * @apiSuccess (Success Response Format) {String} message Response message
                 * @apiSuccess (Success Response Format) {Array} data Array of data
                 *
                 * @apiSuccessExample {json} Success Response
                 {
                   "status": 200,
                   "message": "Sent WhatsApp Chats",
                   "data": [
                        {
                            "account": "+639760713666",
                            "status": "sent",
                            "api": true,
                            "recipient": "+639206150513",
                            "message": "Hello World!",
                            "created": 1645129261
                        },
                        {
                            "account": "+639760713666",
                            "status": "sent",
                            "api": true,
                            "recipient": "+639206150513",
                            "message": "Hello World!",
                            "created": 1645129261
                        },
                        {
                            "account": "+639760713666",
                            "status": "failed",
                            "api": true,
                            "recipient": "+639206150513",
                            "message": "Hello World!",
                            "created": 1645129720
                        }
                    ]
                 }
                 * 
                 * @apiError (Error Response Format) {Number} status List of Codes<br>
                 * 400 = Invalid parameters<br>
                 * 401 = Invalid API secret<br>
                 * 403 = Access denied<br>
                 * 500 = Something went wrong
                 * @apiError (Error Response Format) {String} message Response message
                 * @apiError (Error Response Format) {Array} data Array of data
                 * 
                 * @apiErrorExample {json} Error Response
                 {
                   "status": 400,
                   "message": "Invalid Parameters!",
                   "data": false
                 }
                 *
                 */

                if(!in_array("get_wa_sent", explode(",", $api["permissions"])))
                    response(403, "This API key doesn't have permission to use this endpoint!");

                if(isset($request["page"])):
                    if(!$this->sanitize->isInt($request["page"]))
                        response(400, "Invalid Parameters!");

                    $page = $request["page"];
                else:
                    $page = 1;
                endif;

                if(isset($request["limit"])):
                    if(!$this->sanitize->isInt($request["limit"]))
                        response(400, "Invalid Parameters!");

                    $limit = $request["limit"];
                else:
                    $limit = 10;
                endif;

                $chats = $this->api->getChats($api["uid"], "sent", "sent", abs($page), abs($limit));

                $chatArray = [];

                if(!empty($chats)):
                    foreach($chats as $chat):
                        $account = explode(":", $chat["wid"]);
                        $chatArray[] = [
                            "account" => "+{$account[0]}",
                            "status" => $chat["status"] < 4 ? "sent" : "failed",
                            "api" => $chat["api"] < 2 ? true : false,
                            "recipient" => $chat["phone"],
                            "message" => $chat["message"],
                            "created" => strtotime($chat["create_date"])
                        ];
                    endforeach;
                endif;

                response(200, "Sent WhatsApp Chats", $chatArray);

                break;
            case "wa.pending":
                /**
                 * @api {get} /get/wa.pending Get Pending Chats
                 * @apiName Get Pending Chats
                 * @apiDescription Get Pending Chats. Requires "<strong>get_wa_pending</strong>" API permission.
                 * @apiGroup WhatsApp
                 * @apiVersion 1.0.0
                 *
                 * @apiParam {String} secret The API secret you copied from (Tools -> API Keys) page
                 * @apiParam {Number} [limit=10] Limit the number of results per page.
                 * @apiParam {Number} [page=1] Pagination of results.
                 *
                 * @apiExample {php} PHP Example
                  <?php

                    $apiSecret = "API_SECRET"; // your API secret from (Tools -> API Keys) page

                    $cURL = curl_init();
                    curl_setopt($cURL, CURLOPT_URL, "http://127.0.0.1/zender/api/get/wa.pending?secret={$apiSecret}");
                    curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    $response = curl_exec($cURL);
                    curl_close($cURL);

                    $result = json_decode($response, true);

                    // do something with response
                    print_r($result);
                 * 
                 * @apiExample {python} Python Example
                    import requests
    
                    # your API secret from (Tools -> API Keys) page
                    apiSecret = "API_SECRET"

                    r = requests.get(url = "http://127.0.0.1/zender/api/get/wa.pending", params = {
                        "secret": apiSecret
                    })
                      
                    # do something with response object
                    result = r.json()
                 * 
                 * @apiSuccess (Success Response Format) {Number} status List of Codes
                 * <br> 200 = Success
                 * @apiSuccess (Success Response Format) {String} message Response message
                 * @apiSuccess (Success Response Format) {Array} data Array of data
                 *
                 * @apiSuccessExample {json} Success Response
                 {
                   "status": 200,
                   "message": "Pending WhatsApp Chats",
                   "data": [
                        {
                            "account": "+639760713666",
                            "api": false,
                            "recipient": "+639184661533",
                            "message": "Hello World!",
                            "created": 1645521446
                        },
                        {
                            "account": "+639760713666",
                            "api": true,
                            "recipient": "+639206150513",
                            "message": "Hello World!",
                            "created": 1645521446
                        },
                        {
                            "account": "+639760713666",
                            "api": false,
                            "recipient": "+639184661532",
                            "message": "Hello World!",
                            "created": 1645521446
                        }
                    ]
                 }
                 * 
                 * @apiError (Error Response Format) {Number} status List of Codes<br>
                 * 400 = Invalid parameters<br>
                 * 401 = Invalid API secret<br>
                 * 403 = Access denied<br>
                 * 500 = Something went wrong
                 * @apiError (Error Response Format) {String} message Response message
                 * @apiError (Error Response Format) {Array} data Array of data
                 * 
                 * @apiErrorExample {json} Error Response
                 {
                   "status": 400,
                   "message": "Invalid Parameters!",
                   "data": false
                 }
                 *
                 */

                if(!in_array("get_wa_pending", explode(",", $api["permissions"])))
                    response(403, "This API key doesn't have permission to use this endpoint!");

                if(isset($request["page"])):
                    if(!$this->sanitize->isInt($request["page"]))
                        response(400, "Invalid Parameters!");

                    $page = $request["page"];
                else:
                    $page = 1;
                endif;

                if(isset($request["limit"])):
                    if(!$this->sanitize->isInt($request["limit"]))
                        response(400, "Invalid Parameters!");

                    $limit = $request["limit"];
                else:
                    $limit = 10;
                endif;

                $chats = $this->api->getChats($api["uid"], "sent", "pending", abs($page), abs($limit));

                $chatArray = [];

                if(!empty($chats)):
                    foreach($chats as $chat):
                        $account = explode(":", $chat["wid"]);
                        $chatArray[] = [
                            "account" => "+{$account[0]}",
                            "api" => $chat["api"] < 2 ? true : false,
                            "recipient" => $chat["phone"],
                            "message" => $chat["message"],
                            "created" => strtotime($chat["create_date"])
                        ];
                    endforeach;
                endif;

                response(200, "Pending WhatsApp Chats", $chatArray);

                break;
            case "wa.received":
                /**
                 * @api {get} /get/wa.received Get Received Chats
                 * @apiName Get Received Chats
                 * @apiDescription Get Received Chats. Requires "<strong>get_wa_received</strong>" API permission.
                 * @apiGroup WhatsApp
                 * @apiVersion 1.0.0
                 *
                 * @apiParam {String} secret The API secret you copied from (Tools -> API Keys) page
                 * @apiParam {Number} [limit=10] Limit the number of results per page.
                 * @apiParam {Number} [page=1] Pagination of results.
                 *
                 * @apiExample {php} PHP Example
                  <?php

                    $apiSecret = "API_SECRET"; // your API secret from (Tools -> API Keys) page

                    $cURL = curl_init();
                    curl_setopt($cURL, CURLOPT_URL, "http://127.0.0.1/zender/api/get/wa.received?secret={$apiSecret}");
                    curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    $response = curl_exec($cURL);
                    curl_close($cURL);

                    $result = json_decode($response, true);

                    // do something with response
                    print_r($result);
                 * 
                 * @apiExample {python} Python Example
                    import requests
    
                    # your API secret from (Tools -> API Keys) page
                    apiSecret = "API_SECRET"

                    r = requests.get(url = "http://127.0.0.1/zender/api/get/wa.received", params = {
                        "secret": apiSecret
                    })
                      
                    # do something with response object
                    result = r.json()
                 * 
                 * @apiSuccess (Success Response Format) {Number} status List of Codes
                 * <br> 200 = Success
                 * @apiSuccess (Success Response Format) {String} message Response message
                 * @apiSuccess (Success Response Format) {Array} data Array of data
                 *
                 * @apiSuccessExample {json} Success Response
                 {
                   "status": 200,
                   "message": "Pending WhatsApp Chats",
                   "data": [
                        {
                            "account": "+639760713666",
                            "recipient": "+639184661533",
                            "message": "dnZ2dmI=",
                            "created": 1645232578
                        },
                        {
                            "account": "+639760713666",
                            "recipient": "+639184661533",
                            "message": "gago",
                            "created": 1645232635
                        },
                        {
                            "account": "+639760713666",
                            "recipient": "+639184661533",
                            "message": "hahaha",
                            "created": 1645232650
                        }
                    ]
                 }
                 * 
                 * @apiError (Error Response Format) {Number} status List of Codes<br>
                 * 400 = Invalid parameters<br>
                 * 401 = Invalid API secret<br>
                 * 403 = Access denied<br>
                 * 500 = Something went wrong
                 * @apiError (Error Response Format) {String} message Response message
                 * @apiError (Error Response Format) {Array} data Array of data
                 * 
                 * @apiErrorExample {json} Error Response
                 {
                   "status": 400,
                   "message": "Invalid Parameters!",
                   "data": false
                 }
                 *
                 */

                if(!in_array("get_wa_received", explode(",", $api["permissions"])))
                    response(403, "This API key doesn't have permission to use this endpoint!");

                if(isset($request["page"])):
                    if(!$this->sanitize->isInt($request["page"]))
                        response(400, "Invalid Parameters!");

                    $page = $request["page"];
                else:
                    $page = 1;
                endif;

                if(isset($request["limit"])):
                    if(!$this->sanitize->isInt($request["limit"]))
                        response(400, "Invalid Parameters!");

                    $limit = $request["limit"];
                else:
                    $limit = 10;
                endif;

                $chats = $this->api->getChats($api["uid"], "received", false, abs($page), abs($limit));

                $chatArray = [];

                if(!empty($chats)):
                    foreach($chats as $chat):
                        $account = explode(":", $chat["wid"]);
                        $chatArray[] = [
                            "account" => "+{$account[0]}",
                            "recipient" => $chat["phone"],
                            "message" => $chat["message"],
                            "created" => strtotime($chat["receive_date"])
                        ];
                    endforeach;
                endif;

                response(200, "Received WhatsApp Chats", $chatArray);

                break;
            case "contacts":
                /**
                 * @api {get} /get/contacts Get Contacts
                 * @apiName Get Contacts
                 * @apiDescription Get Contacts. Requires "<strong>get_contacts</strong>" API permission.
                 * @apiGroup Contacts
                 * @apiVersion 1.0.0
                 *
                 * @apiParam {String} secret The API secret you copied from (Tools -> API Keys) page
                 * @apiParam {Number} [limit=10] Limit the number of results per page.
                 * @apiParam {Number} [page=1] Pagination of results.
                 *
                 * @apiExample {php} PHP Example
                  <?php

                    $apiSecret = "API_SECRET"; // your API secret from (Tools -> API Keys) page

                    $cURL = curl_init();
                    curl_setopt($cURL, CURLOPT_URL, "http://127.0.0.1/zender/api/get/contacts?secret={$apiSecret}");
                    curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    $response = curl_exec($cURL);
                    curl_close($cURL);

                    $result = json_decode($response, true);

                    // do something with response
                    print_r($result);
                 * 
                 * @apiExample {python} Python Example
                    import requests
    
                    # your API secret from (Tools -> API Keys) page
                    apiSecret = "API_SECRET"

                    r = requests.get(url = "http://127.0.0.1/zender/api/get/contacts", params = {
                        "secret": apiSecret
                    })
                      
                    # do something with response object
                    result = r.json()
                 * 
                 * @apiSuccess (Success Response Format) {Number} status List of Codes
                 * <br> 200 = Success
                 * @apiSuccess (Success Response Format) {String} message Response message
                 * @apiSuccess (Success Response Format) {Array} data Array of data
                 *
                 * @apiSuccessExample {json} Success Response
                 {
                   "status": 200,
                   "message": "Saved Contacts",
                   "data": [
                        {
                            "id": 2,
                            "groups": [
                                "1"
                            ],
                            "phone": "+639184661538",
                            "name": "Shane"
                        },
                        {
                            "id": 3,
                            "groups": [
                                "1",
                                "9",
                                "10",
                                "11"
                            ],
                            "phone": "+639206150514",
                            "name": "Terry Bom"
                        },
                        {
                            "id": 4,
                            "groups": [
                                "1",
                                "9"
                            ],
                            "phone": "+639184661532",
                            "name": "Jake Thrower"
                        }
                    ]
                 }
                 * 
                 * @apiError (Error Response Format) {Number} status List of Codes<br>
                 * 400 = Invalid parameters<br>
                 * 401 = Invalid API secret<br>
                 * 403 = Access denied<br>
                 * 500 = Something went wrong
                 * @apiError (Error Response Format) {String} message Response message
                 * @apiError (Error Response Format) {Array} data Array of data
                 * 
                 * @apiErrorExample {json} Error Response
                 {
                   "status": 400,
                   "message": "Invalid Parameters!",
                   "data": false
                 }
                 *
                 */

                if(!in_array("get_contacts", explode(",", $api["permissions"])))
                    response(403, "This API key doesn't have permission to use this endpoint!");

                if(isset($request["page"])):
                    if(!$this->sanitize->isInt($request["page"]))
                        response(400, "Invalid Parameters!");

                    $page = $request["page"];
                else:
                    $page = 1;
                endif;

                if(isset($request["limit"])):
                    if(!$this->sanitize->isInt($request["limit"]))
                        response(400, "Invalid Parameters!");

                    $limit = $request["limit"];
                else:
                    $limit = 10;
                endif;

                $contacts = $this->api->getContacts($api["uid"], abs($page), abs($limit));

                $contactArray = [];

                if(!empty($contacts)):
                    foreach($contacts as $contact):
                        $groups = explode(",", $contact["groups"]);
                        $contactArray[] = [
                            "id" => (int) $contact["id"],
                            "groups" => is_array($groups) ? $groups : [],
                            "phone" => $contact["phone"],
                            "name" => $contact["name"]
                        ];
                    endforeach;
                endif;

                response(200, "Saved Contacts", $contactArray);

                break;
            case "groups":
                /**
                 * @api {get} /get/groups Get Groups
                 * @apiName Get Groups
                 * @apiDescription Get Groups. Requires "<strong>get_groups</strong>" API permission.
                 * @apiGroup Contacts
                 * @apiVersion 1.0.0
                 *
                 * @apiParam {String} secret The API secret you copied from (Tools -> API Keys) page
                 * @apiParam {Number} [limit=10] Limit the number of results per page.
                 * @apiParam {Number} [page=1] Pagination of results.
                 *
                 * @apiExample {php} PHP Example
                  <?php

                    $apiSecret = "API_SECRET"; // your API secret from (Tools -> API Keys) page

                    $cURL = curl_init();
                    curl_setopt($cURL, CURLOPT_URL, "http://127.0.0.1/zender/api/get/groups?secret={$apiSecret}");
                    curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    $response = curl_exec($cURL);
                    curl_close($cURL);

                    $result = json_decode($response, true);

                    // do something with response
                    print_r($result);
                 * 
                 * @apiExample {python} Python Example
                    import requests
    
                    # your API secret from (Tools -> API Keys) page
                    apiSecret = "API_SECRET"

                    r = requests.get(url = "http://127.0.0.1/zender/api/get/groups", params = {
                        "secret": apiSecret
                    })
                      
                    # do something with response object
                    result = r.json()
                 * 
                 * @apiSuccess (Success Response Format) {Number} status List of Codes
                 * <br> 200 = Success
                 * @apiSuccess (Success Response Format) {String} message Response message
                 * @apiSuccess (Success Response Format) {Array} data Array of data
                 *
                 * @apiSuccessExample {json} Success Response
                 {
                   "status": 200,
                   "message": "Contact Groups",
                   "data": [
                        {
                            "id": 1,
                            "name": "Default"
                        },
                        {
                            "id": 9,
                            "name": "Happy Group"
                        },
                        {
                            "id": 10,
                            "name": "Riders"
                        }
                    ]
                 }
                 * 
                 * @apiError (Error Response Format) {Number} status List of Codes<br>
                 * 400 = Invalid parameters<br>
                 * 401 = Invalid API secret<br>
                 * 403 = Access denied<br>
                 * 500 = Something went wrong
                 * @apiError (Error Response Format) {String} message Response message
                 * @apiError (Error Response Format) {Array} data Array of data
                 * 
                 * @apiErrorExample {json} Error Response
                 {
                   "status": 400,
                   "message": "Invalid Parameters!",
                   "data": false
                 }
                 *
                 */

                if(!in_array("get_groups", explode(",", $api["permissions"])))
                    response(403, "This API key doesn't have permission to use this endpoint!");

                if(isset($request["page"])):
                    if(!$this->sanitize->isInt($request["page"]))
                        response(400, "Invalid Parameters!");

                    $page = $request["page"];
                else:
                    $page = 1;
                endif;

                if(isset($request["limit"])):
                    if(!$this->sanitize->isInt($request["limit"]))
                        response(400, "Invalid Parameters!");

                    $limit = $request["limit"];
                else:
                    $limit = 10;
                endif;

                $groups = $this->api->getGroups($api["uid"], abs($page), abs($limit));

                $groupArray = [];

                if(!empty($groups)):
                    foreach($groups as $group):
                        $groupArray[] = [
                            "id" => (int) $group["id"],
                            "name" => $group["name"]
                        ];
                    endforeach;
                endif;

                response(200, "Contact Groups", $groupArray);

                break;
            case "unsubscribed":
                /**
                 * @api {get} /get/unsubscribed Get Unsubscribed
                 * @apiName Get Unsubscribed
                 * @apiDescription Get Unsubscribed Contacts. Requires "<strong>get_unsubscribed</strong>" API permission.
                 * @apiGroup Contacts
                 * @apiVersion 1.0.0
                 *
                 * @apiParam {String} secret The API secret you copied from (Tools -> API Keys) page
                 * @apiParam {Number} [limit=10] Limit the number of results per page.
                 * @apiParam {Number} [page=1] Pagination of results.
                 *
                 * @apiExample {php} PHP Example
                  <?php

                    $apiSecret = "API_SECRET"; // your API secret from (Tools -> API Keys) page

                    $cURL = curl_init();
                    curl_setopt($cURL, CURLOPT_URL, "http://127.0.0.1/zender/api/get/unsubscribed?secret={$apiSecret}");
                    curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    $response = curl_exec($cURL);
                    curl_close($cURL);

                    $result = json_decode($response, true);

                    // do something with response
                    print_r($result);
                 * 
                 * @apiExample {python} Python Example
                    import requests
    
                    # your API secret from (Tools -> API Keys) page
                    apiSecret = "API_SECRET"

                    r = requests.get(url = "http://127.0.0.1/zender/api/get/unsubscribed", params = {
                        "secret": apiSecret
                    })
                      
                    # do something with response object
                    result = r.json()
                 * 
                 * @apiSuccess (Success Response Format) {Number} status List of Codes
                 * <br> 200 = Success
                 * @apiSuccess (Success Response Format) {String} message Response message
                 * @apiSuccess (Success Response Format) {Array} data Array of data
                 *
                 * @apiSuccessExample {json} Success Response
                 {
                   "status": 200,
                   "message": "Unsubscribed Contacts",
                   "data": [
                        {
                            "id": 2,
                            "phone": "+639694967617",
                            "created": 1645755138
                        }
                    ]
                 }
                 * 
                 * @apiError (Error Response Format) {Number} status List of Codes<br>
                 * 400 = Invalid parameters<br>
                 * 401 = Invalid API secret<br>
                 * 403 = Access denied<br>
                 * 500 = Something went wrong
                 * @apiError (Error Response Format) {String} message Response message
                 * @apiError (Error Response Format) {Array} data Array of data
                 * 
                 * @apiErrorExample {json} Error Response
                 {
                   "status": 400,
                   "message": "Invalid Parameters!",
                   "data": false
                 }
                 *
                 */

                if(!in_array("get_unsubscribed", explode(",", $api["permissions"])))
                    response(403, "This API key doesn't have permission to use this endpoint!");

                if(isset($request["page"])):
                    if(!$this->sanitize->isInt($request["page"]))
                        response(400, "Invalid Parameters!");

                    $page = $request["page"];
                else:
                    $page = 1;
                endif;

                if(isset($request["limit"])):
                    if(!$this->sanitize->isInt($request["limit"]))
                        response(400, "Invalid Parameters!");

                    $limit = $request["limit"];
                else:
                    $limit = 10;
                endif;

                $unsubscribed = $this->api->getUnsubscribed($api["uid"], abs($page), abs($limit));

                $unsubscribeArray = [];

                if(!empty($unsubscribed)):
                    foreach($unsubscribed as $unsubscribe):
                        $unsubscribeArray[] = [
                            "id" => (int) $unsubscribe["id"],
                            "phone" => $unsubscribe["phone"],
                            "created" => strtotime($unsubscribe["create_date"])
                        ];
                    endforeach;
                endif;

                response(200, "Unsubscribed Contacts", $unsubscribeArray);

                break;
            case "ussd":
                /**
                 * @api {get} /get/ussd Get USSD Requests
                 * @apiName Get USSD Requests
                 * @apiDescription Get USSD Requests. Requires "<strong>get_ussd</strong>" API permission.
                 * @apiGroup USSD
                 * @apiVersion 1.0.0
                 *
                 * @apiParam {String} secret The API secret you copied from (Tools -> API Keys) page
                 * @apiParam {Number} [limit=10] Limit the number of results per page.
                 * @apiParam {Number} [page=1] Pagination of results.
                 *
                 * @apiExample {php} PHP Example
                  <?php

                    $apiSecret = "API_SECRET"; // your API secret from (Tools -> API Keys) page

                    $cURL = curl_init();
                    curl_setopt($cURL, CURLOPT_URL, "http://127.0.0.1/zender/api/get/ussd?secret={$apiSecret}");
                    curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    $response = curl_exec($cURL);
                    curl_close($cURL);

                    $result = json_decode($response, true);

                    // do something with response
                    print_r($result);
                 * 
                 * @apiExample {python} Python Example
                    import requests
    
                    # your API secret from (Tools -> API Keys) page
                    apiSecret = "API_SECRET"

                    r = requests.get(url = "http://127.0.0.1/zender/api/get/ussd", params = {
                        "secret": apiSecret
                    })
                      
                    # do something with response object
                    result = r.json()
                 * 
                 * @apiSuccess (Success Response Format) {Number} status List of Codes
                 * <br> 200 = Success
                 * @apiSuccess (Success Response Format) {String} message Response message
                 * @apiSuccess (Success Response Format) {Array} data Array of data
                 *
                 * @apiSuccessExample {json} Success Response
                 {
                   "status": 200,
                   "message": "USSD Requests",
                   "data": [
                        {
                            "id": 5,
                            "device": "00000000-0000-0000-d57d-f30cb6a89289",
                            "sim": 1,
                            "code": "*143#",
                            "response": "Sorry! You are not allowed to use this service.",
                            "status": "completed",
                            "created": 1645043019
                        },
                        {
                            "id": 6,
                            "device": "00000000-0000-0000-d57d-f30cb6a89289",
                            "sim": 1,
                            "code": "*145#",
                            "response": "Your balance is 14.60",
                            "status": "completed",
                            "created": 1645043024
                        },
                        {
                            "id": 13,
                            "device": "00000000-0000-0000-d57d-f30cb6a89289",
                            "sim": 2,
                            "code": "*121#",
                            "response": "Sorry! Invalid MMI Code.",
                            "status": "completed",
                            "created": 1645413608
                        }
                    ]
                 }
                 * 
                 * @apiError (Error Response Format) {Number} status List of Codes<br>
                 * 400 = Invalid parameters<br>
                 * 401 = Invalid API secret<br>
                 * 403 = Access denied<br>
                 * 500 = Something went wrong
                 * @apiError (Error Response Format) {String} message Response message
                 * @apiError (Error Response Format) {Array} data Array of data
                 * 
                 * @apiErrorExample {json} Error Response
                 {
                   "status": 400,
                   "message": "Invalid Parameters!",
                   "data": false
                 }
                 *
                 */

                if(!in_array("get_ussd", explode(",", $api["permissions"])))
                    response(403, "This API key doesn't have permission to use this endpoint!");

                if(isset($request["page"])):
                    if(!$this->sanitize->isInt($request["page"]))
                        response(400, "Invalid Parameters!");

                    $page = $request["page"];
                else:
                    $page = 1;
                endif;

                if(isset($request["limit"])):
                    if(!$this->sanitize->isInt($request["limit"]))
                        response(400, "Invalid Parameters!");

                    $limit = $request["limit"];
                else:
                    $limit = 10;
                endif;

                $ussds = $this->api->getUssd($api["uid"], abs($page), abs($limit));

                $ussdArray = [];

                if(!empty($ussds)):
                    foreach($ussds as $ussd):
                        $ussdArray[] = [
                            "id" => (int) $ussd["id"],
                            "device" => $ussd["did"],
                            "sim" => $ussd["sim"] < 1 ? 1 : 2,
                            "code" => $ussd["code"],
                            "response" => $ussd["response"],
                            "status" => $ussd["status"] < 2 ? "pending" : ($ussd["status"] == 2 ? "queued" : "completed"),
                            "created" => strtotime($ussd["create_date"])
                        ];
                    endforeach;
                endif;

                response(200, "USSD Requests", $ussdArray);

                break;
            case "notifications":
                /**
                 * @api {get} /get/notifications Get Notifications
                 * @apiName Get Notifications
                 * @apiDescription Get Notifications. Requires "<strong>get_notifications</strong>" API permission.
                 * @apiGroup Notifications
                 * @apiVersion 1.0.0
                 *
                 * @apiParam {String} secret The API secret you copied from (Tools -> API Keys) page
                 * @apiParam {Number} [limit=10] Limit the number of results per page.
                 * @apiParam {Number} [page=1] Pagination of results.
                 *
                 * @apiExample {php} PHP Example
                  <?php

                    $apiSecret = "API_SECRET"; // your API secret from (Tools -> API Keys) page

                    $cURL = curl_init();
                    curl_setopt($cURL, CURLOPT_URL, "http://127.0.0.1/zender/api/get/notifications?secret={$apiSecret}");
                    curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    $response = curl_exec($cURL);
                    curl_close($cURL);

                    $result = json_decode($response, true);

                    // do something with response
                    print_r($result);
                 * 
                 * @apiExample {python} Python Example
                    import requests
    
                    # your API secret from (Tools -> API Keys) page
                    apiSecret = "API_SECRET"

                    r = requests.get(url = "http://127.0.0.1/zender/api/get/notifications", params = {
                        "secret": apiSecret
                    })
                      
                    # do something with response object
                    result = r.json()
                 * 
                 * @apiSuccess (Success Response Format) {Number} status List of Codes
                 * <br> 200 = Success
                 * @apiSuccess (Success Response Format) {String} message Response message
                 * @apiSuccess (Success Response Format) {Array} data Array of data
                 *
                 * @apiSuccessExample {json} Success Response
                 {
                   "status": 200,
                   "message": "Received Notifications",
                   "data": [
                        {
                            "id": 1,
                            "device": "00000000-0000-0000-d57d-f30cb6a89289",
                            "package_name": "com.facebook.orca",
                            "title": "Darren Shmuck",
                            "content": "Hello World!",
                            "timestamp": 1645052535
                        },
                        {
                            "id": 2,
                            "device": "00000000-0000-0000-d57d-f30cb6a89289",
                            "package_name": "com.facebook.katana",
                            "title": "Michael shared your post",
                            "content": "Michael shared your post",
                            "timestamp": 1645052541
                        },
                        {
                            "id": 3,
                            "device": "00000000-0000-0000-d57d-f30cb6a89289",
                            "package_name": "com.facebook.orca",
                            "title": "Shane Blake",
                            "content": "Hello World!",
                            "timestamp": 1645052543
                        }
                    ]
                 }
                 * 
                 * @apiError (Error Response Format) {Number} status List of Codes<br>
                 * 400 = Invalid parameters<br>
                 * 401 = Invalid API secret<br>
                 * 403 = Access denied<br>
                 * 500 = Something went wrong
                 * @apiError (Error Response Format) {String} message Response message
                 * @apiError (Error Response Format) {Array} data Array of data
                 * 
                 * @apiErrorExample {json} Error Response
                 {
                   "status": 400,
                   "message": "Invalid Parameters!",
                   "data": false
                 }
                 *
                 */

                if(!in_array("get_notifications", explode(",", $api["permissions"])))
                    response(403, "This API key doesn't have permission to use this endpoint!");

                if(isset($request["page"])):
                    if(!$this->sanitize->isInt($request["page"]))
                        response(400, "Invalid Parameters!");

                    $page = $request["page"];
                else:
                    $page = 1;
                endif;

                if(isset($request["limit"])):
                    if(!$this->sanitize->isInt($request["limit"]))
                        response(400, "Invalid Parameters!");

                    $limit = $request["limit"];
                else:
                    $limit = 10;
                endif;

                $notifications = $this->api->getNotifications($api["uid"], abs($page), abs($limit));

                $notificationArray = [];

                if(!empty($notifications)):
                    foreach($notifications as $notification):
                        $notificationArray[] = [
                            "id" => (int) $notification["id"],
                            "device" => $notification["did"],
                            "package_name" => $notification["package"],
                            "title" => $notification["title"],
                            "content" => $notification["text"],
                            "timestamp" => strtotime($notification["create_date"])
                        ];
                    endforeach;
                endif;

                response(200, "Received Notifications", $notificationArray);

                break;
            case "wa.accounts":
                /**
                 * @api {get} /get/wa.accounts Get Accounts
                 * @apiName Get Accounts
                 * @apiDescription Get Accounts. Requires "<strong>get_wa_accounts</strong>" API permission.
                 * @apiGroup WhatsApp
                 * @apiVersion 1.0.0
                 *
                 * @apiParam {String} secret The API secret you copied from (Tools -> API Keys) page
                 * @apiParam {Number} [limit=10] Limit the number of results per page.
                 * @apiParam {Number} [page=1] Pagination of results.
                 *
                 * @apiExample {php} PHP Example
                  <?php

                    $apiSecret = "API_SECRET"; // your API secret from (Tools -> API Keys) page

                    $cURL = curl_init();
                    curl_setopt($cURL, CURLOPT_URL, "http://127.0.0.1/zender/api/get/wa.accounts?secret={$apiSecret}");
                    curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    $response = curl_exec($cURL);
                    curl_close($cURL);

                    $result = json_decode($response, true);

                    // do something with response
                    print_r($result);
                 * 
                 * @apiExample {python} Python Example
                    import requests
    
                    # your API secret from (Tools -> API Keys) page
                    apiSecret = "API_SECRET"

                    r = requests.get(url = "http://127.0.0.1/zender/api/get/wa.accounts", params = {
                        "secret": apiSecret
                    })
                      
                    # do something with response object
                    result = r.json()
                 * 
                 * @apiSuccess (Success Response Format) {Number} status List of Codes
                 * <br> 200 = Success
                 * @apiSuccess (Success Response Format) {String} message Response message
                 * @apiSuccess (Success Response Format) {Array} data Array of data
                 *
                 * @apiSuccessExample {json} Success Response
                 {
                   "status": 200,
                   "message": "WhatsApp Accounts",
                   "data": [
                        {
                            "id": "1",
                            "phone": "+639760713666",
                            "unique": "90cf7d40a467d5f40a39fca222c216449cb9abee73e5e2b0b321060c2ae06a8fa9e45486",
                            "created": 1645128758
                        }
                    ]
                 }
                 * 
                 * @apiError (Error Response Format) {Number} status List of Codes<br>
                 * 400 = Invalid parameters<br>
                 * 401 = Invalid API secret<br>
                 * 403 = Access denied<br>
                 * 500 = Something went wrong
                 * @apiError (Error Response Format) {String} message Response message
                 * @apiError (Error Response Format) {Array} data Array of data
                 * 
                 * @apiErrorExample {json} Error Response
                 {
                   "status": 400,
                   "message": "Invalid Parameters!",
                   "data": false
                 }
                 *
                 */

                if(!in_array("get_wa_accounts", explode(",", $api["permissions"])))
                    response(403, "This API key doesn't have permission to use this endpoint!");

                if(isset($request["page"])):
                    if(!$this->sanitize->isInt($request["page"]))
                        response(400, "Invalid Parameters!");

                    $page = $request["page"];
                else:
                    $page = 1;
                endif;

                if(isset($request["limit"])):
                    if(!$this->sanitize->isInt($request["limit"]))
                        response(400, "Invalid Parameters!");

                    $limit = $request["limit"];
                else:
                    $limit = 10;
                endif;

                $accounts = $this->api->getWaAccounts($api["uid"], abs($page), abs($limit));

                $accountArray = [];

                if(!empty($accounts)):
                    foreach($accounts as $account):
                        $phone = explode(":", $account["wid"]);
                        $accountArray[] = [
                            "id" => $account["id"],
                            "phone" => "+{$phone[0]}",
                            "unique" => $account["unique"],
                            "created" => strtotime($account["create_date"])
                        ];
                    endforeach;
                endif;

                response(200, "WhatsApp Accounts", $accountArray);

                break;
            case "devices":
                /**
                 * @api {get} /get/devices Get Devices
                 * @apiName Get Devices
                 * @apiDescription Get Devices. Requires "<strong>get_devices</strong>" API permission.
                 * @apiGroup SMS
                 * @apiVersion 1.0.0
                 *
                 * @apiParam {String} secret The API secret you copied from (Tools -> API Keys) page
                 * @apiParam {Number} [limit=10] Limit the number of results per page.
                 * @apiParam {Number} [page=1] Pagination of results.
                 *
                 * @apiExample {php} PHP Example
                  <?php

                    $apiSecret = "API_SECRET"; // your API secret from (Tools -> API Keys) page

                    $cURL = curl_init();
                    curl_setopt($cURL, CURLOPT_URL, "http://127.0.0.1/zender/api/get/devices?secret={$apiSecret}");
                    curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    $response = curl_exec($cURL);
                    curl_close($cURL);

                    $result = json_decode($response, true);

                    // do something with response
                    print_r($result);
                 * 
                 * @apiExample {python} Python Example
                    import requests
    
                    # your API secret from (Tools -> API Keys) page
                    apiSecret = "API_SECRET"

                    r = requests.get(url = "http://127.0.0.1/zender/api/get/devices", params = {
                        "secret": apiSecret
                    })
                      
                    # do something with response object
                    result = r.json()
                 * 
                 * @apiSuccess (Success Response Format) {Number} status List of Codes
                 * <br> 200 = Success
                 * @apiSuccess (Success Response Format) {String} message Response message
                 * @apiSuccess (Success Response Format) {Array} data Array of data
                 *
                 * @apiSuccessExample {json} Success Response
                 {
                   "status": 200,
                   "message": "Android Devices",
                   "data": [
                        {
                            "id": "49",
                            "unique": "00000000-0000-0000-d57d-f30cb6a89289",
                            "name": "F11 Phone",
                            "version": "Android 11",
                            "manufacturer": "OPPO",
                            "random_send": false,
                            "random_min": 5,
                            "random_max": 10,
                            "partner": false,
                            "partner_sim": [
                                "2"
                            ],
                            "partner_priority": false,
                            "partner_country": "PH",
                            "partner_rate": 5,
                            "partner_currency": "PHP",
                            "created": 1636462504
                        }
                    ]
                 }
                 * 
                 * @apiError (Error Response Format) {Number} status List of Codes<br>
                 * 400 = Invalid parameters<br>
                 * 401 = Invalid API secret<br>
                 * 403 = Access denied<br>
                 * 500 = Something went wrong
                 * @apiError (Error Response Format) {String} message Response message
                 * @apiError (Error Response Format) {Array} data Array of data
                 * 
                 * @apiErrorExample {json} Error Response
                 {
                   "status": 400,
                   "message": "Invalid Parameters!",
                   "data": false
                 }
                 *
                 */

                if(!in_array("get_devices", explode(",", $api["permissions"])))
                    response(403, "This API key doesn't have permission to use this endpoint!");

                if(isset($request["page"])):
                    if(!$this->sanitize->isInt($request["page"]))
                        response(400, "Invalid Parameters!");

                    $page = $request["page"];
                else:
                    $page = 1;
                endif;

                if(isset($request["limit"])):
                    if(!$this->sanitize->isInt($request["limit"]))
                        response(400, "Invalid Parameters!");

                    $limit = $request["limit"];
                else:
                    $limit = 10;
                endif;

                $devices = $this->api->getDevices($api["uid"], abs($page), abs($limit));

                $deviceArray = [];

                if(!empty($devices)):
                    foreach($devices as $device):
                        switch(round((int) $device["version"])):
                            case 4:
                                $android = "Android KitKat";
                                break;
                            case 5:
                                $android = "Android Lollipop";
                                break;
                            case 6:
                                $android = "Android Marshmallow";
                                break;
                            case 7:
                                $android = "Android Nougat";
                                break;
                            case 8:
                                $android = "Android Oreo";
                                break;
                            case 9:
                                $android = "Android Pie";
                                break;
                            case 10:
                                $android = "Android 10";
                                break;
                            case 11:
                                $android = "Android 11";
                                break;
                            case 12:
                                $android = "Android 12";
                                break;
                            default:
                                $android = "Unknown";
                        endswitch;

                        $slots = explode(",", $device["global_slots"]);
                        $currency = country($device["country"])->getCurrency()["iso_4217_code"];

                        $deviceArray[] = [
                            "id" => $device["id"],
                            "unique" => $device["did"],
                            "name" => $device["name"],
                            "version" => $android,
                            "manufacturer" => $device["manufacturer"],
                            "random_send" => $device["random_send"] < 2 ? true : false,
                            "random_min" => (int) $device["random_min"],
                            "random_max" => (int) $device["random_max"],
                            "partner" => $device["global_device"] < 2 ? true : false,
                            "partner_sim" => is_array($slots) ? $slots : [],
                            "partner_priority" => $device["global_priority"] < 2 ? true : false,
                            "partner_country" => strtoupper($device["country"]),
                            "partner_rate" => (float) $device["rate"],
                            "partner_currency" => strtoupper($currency),
                            "created" => strtotime($device["create_date"])
                        ];
                    endforeach;
                endif;

                response(200, "Android Devices", $deviceArray);

                break;
            case "rates":
                /**
                 * @api {get} /get/rates Get Gateway Rates
                 * @apiName Get Gateway Rates
                 * @apiDescription Get Gateway Rates. Requires "<strong>get_rates</strong>" API permission.
                 * @apiGroup System
                 * @apiVersion 1.0.0
                 *
                 * @apiParam {String} secret The API secret you copied from (Tools -> API Keys) page
                 *
                 * @apiExample {php} PHP Example
                  <?php

                    $apiSecret = "API_SECRET"; // your API secret from (Tools -> API Keys) page

                    $cURL = curl_init();
                    curl_setopt($cURL, CURLOPT_URL, "http://127.0.0.1/zender/api/get/rates?secret={$apiSecret}");
                    curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    $response = curl_exec($cURL);
                    curl_close($cURL);

                    $result = json_decode($response, true);

                    // do something with response
                    print_r($result);
                 * 
                 * @apiExample {python} Python Example
                    import requests
    
                    # your API secret from (Tools -> API Keys) page
                    apiSecret = "API_SECRET"

                    r = requests.get(url = "http://127.0.0.1/zender/api/get/rates", params = {
                        "secret": apiSecret
                    })
                      
                    # do something with response object
                    result = r.json()
                 * 
                 * @apiSuccess (Success Response Format) {Number} status List of Codes
                 * <br> 200 = Success
                 * @apiSuccess (Success Response Format) {String} message Response message
                 * @apiSuccess (Success Response Format) {Array} data Array of data
                 *
                 * @apiSuccessExample {json} Success Response
                 {
                   "status": 200,
                   "message": "Gateway Rates",
                   "data": {
                        "gateways": [
                            {
                                "id": 1,
                                "name": "Twilio",
                                "currency": "GBP",
                                "pricing": {
                                    "default": "0.01",
                                    "countries": {
                                        "us": "0.01",
                                        "ph": "10",
                                        "gb": "0.02"
                                    }
                                }
                            }
                        ],
                        "partners": [
                            {
                                "unique": "00000000-0000-0000-d57d-f30cb6a89289",
                                "name": "F11 Phone",
                                "version": "Android 11",
                                "priority": false,
                                "sim": [
                                    "2"
                                ],
                                "country": "PH",
                                "currency": "PHP",
                                "rate": 5,
                                "owner": "mail@owneremail.com",
                                "status": "online"
                            }
                        ]
                    }
                 }
                 * 
                 * @apiError (Error Response Format) {Number} status List of Codes<br>
                 * 400 = Invalid parameters<br>
                 * 401 = Invalid API secret<br>
                 * 403 = Access denied<br>
                 * 500 = Something went wrong
                 * @apiError (Error Response Format) {String} message Response message
                 * @apiError (Error Response Format) {Array} data Array of data
                 * 
                 * @apiErrorExample {json} Error Response
                 {
                   "status": 400,
                   "message": "Invalid Parameters!",
                   "data": false
                 }
                 *
                 */

                if(!in_array("get_rates", explode(",", $api["permissions"])))
                    response(403, "This API key doesn't have permission to use this endpoint!");

                $gateways = $this->api->getGateways();

                $gatewayArray = [];

                if(!empty($gateways)):
                    foreach($gateways as $gateway):
                        $gatewayArray[] = [
                            "id" => (int) $gateway["id"],
                            "name" => $gateway["name"],
                            "currency" => strtoupper(system_currency),
                            "pricing" => json_decode($gateway["pricing"], true)
                        ];
                    endforeach;
                endif;

                $partners = $this->api->getPartners($api["uid"]);

                $partnerArray = [];

                if(!empty($partners)):
                    foreach($partners as $partner):
                        switch(round((int) $partner["version"])):
                            case 4:
                                $android = "Android KitKat";
                                break;
                            case 5:
                                $android = "Android Lollipop";
                                break;
                            case 6:
                                $android = "Android Marshmallow";
                                break;
                            case 7:
                                $android = "Android Nougat";
                                break;
                            case 8:
                                $android = "Android Oreo";
                                break;
                            case 9:
                                $android = "Android Pie";
                                break;
                            case 10:
                                $android = "Android 10";
                                break;
                            case 11:
                                $android = "Android 11";
                                break;
                            case 12:
                                $android = "Android 12";
                                break;
                            default:
                                $android = "Unknown";
                        endswitch;

                        $slots = explode(",", $partner["slots"]);
                        $currency = country($partner["country"])->getCurrency()["iso_4217_code"];

                        $partnerArray[] = [
                            "unique" => $partner["unique"],
                            "name" => $partner["name"],
                            "version" => $android,
                            "priority" => $partner["priority"] < 2 ? true : false,
                            "sim" => $slots,
                            "country" => strtoupper($partner["country"]),
                            "currency" => strtoupper($currency),
                            "rate" => (int) $partner["rate"],
                            "owner" => $partner["owner"],
                            "status" => $partner["status"] < 2 ? "online" : "offline"
                        ];
                    endforeach;
                endif;

                response(200, "Gateway Rates", [
                    "gateways" => $gatewayArray,
                    "partners" => $partnerArray
                ]);

                break;
            case "shorteners":
                /**
                 * @api {get} /get/shorteners Get Shorteners
                 * @apiName Get Shorteners
                 * @apiDescription Get Shorteners. Requires "<strong>get_shorteners</strong>" API permission.
                 * @apiGroup System
                 * @apiVersion 1.0.0
                 *
                 * @apiParam {String} secret The API secret you copied from (Tools -> API Keys) page
                 *
                 * @apiExample {php} PHP Example
                  <?php

                    $apiSecret = "API_SECRET"; // your API secret from (Tools -> API Keys) page

                    $cURL = curl_init();
                    curl_setopt($cURL, CURLOPT_URL, "http://127.0.0.1/zender/api/get/shorteners?secret={$apiSecret}");
                    curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    $response = curl_exec($cURL);
                    curl_close($cURL);

                    $result = json_decode($response, true);

                    // do something with response
                    print_r($result);
                 * 
                 * @apiExample {python} Python Example
                    import requests
    
                    # your API secret from (Tools -> API Keys) page
                    apiSecret = "API_SECRET"

                    r = requests.get(url = "http://127.0.0.1/zender/api/get/shorteners", params = {
                        "secret": apiSecret
                    })
                      
                    # do something with response object
                    result = r.json()
                 * 
                 * @apiSuccess (Success Response Format) {Number} status List of Codes
                 * <br> 200 = Success
                 * @apiSuccess (Success Response Format) {String} message Response message
                 * @apiSuccess (Success Response Format) {Array} data Array of data
                 *
                 * @apiSuccessExample {json} Success Response
                 {
                   "status": 200,
                   "message": "Available Shorteners",
                   "data": [
                        {
                            "id": 1,
                            "name": "Bitly"
                        }
                    ]
                 }
                 * 
                 * @apiError (Error Response Format) {Number} status List of Codes<br>
                 * 400 = Invalid parameters<br>
                 * 401 = Invalid API secret<br>
                 * 403 = Access denied<br>
                 * 500 = Something went wrong
                 * @apiError (Error Response Format) {String} message Response message
                 * @apiError (Error Response Format) {Array} data Array of data
                 * 
                 * @apiErrorExample {json} Error Response
                 {
                   "status": 400,
                   "message": "Invalid Parameters!",
                   "data": false
                 }
                 *
                 */

                if(!in_array("get_shorteners", explode(",", $api["permissions"])))
                    response(403, "This API key doesn't have permission to use this endpoint!");

                $shorteners = $this->api->getShorteners($api["uid"]);

                $shortenerArray = [];

                if(!empty($shorteners)):
                    foreach($shorteners as $shortener):
                        $shortenerArray[] = [
                            "id" => (int) $shortener["id"],
                            "name" => $shortener["name"],
                        ];
                    endforeach;
                endif;

                response(200, "Available Shorteners", $shortenerArray);

                break;
            default:
                response(400, "Invalid API Endpoint!");
        endswitch;
	}

	public function create()
	{
        $this->header->allow();

        $this->cache->container("system.settings");

        if($this->cache->empty()):
            $this->cache->setArray($this->system->getSettings());
        endif;

        set_system($this->cache->getAll());

        $request = $this->sanitize->array($_REQUEST);
        $type = $this->sanitize->string($this->url->segment(4));

        if(!isset($request["secret"]))
            response(400, "Invalid Parameters!");

        if($this->api->checkApikey($request["secret"]) < 1)
            response(401, "Invalid API secret supplied!");

        $api = $this->api->getApikey($request["secret"]);

        $permissions = explode(",", $api["permissions"]);

        if(!is_array($permissions))
            response(403, "Insufficient Permissions!");

        $this->echo->_cache = $this->cache;
        $this->echo->_guzzle = $this->guzzle;

        switch($type):
            case "contact":
                /**
                 * @api {post} /create/contact Create Contact
                 * @apiName Create Contact
                 * @apiDescription Create Contact. Requires "<strong>create_contact</strong>" API permission.
                 * @apiGroup Contacts
                 * @apiVersion 1.0.0
                 *
                 * @apiParam {String} secret The API secret you copied from (Tools -> API Keys) page
                 * @apiParam {String} phone phone Recipient mobile number, it will accept E.164 formatted number or locally formatted numbers using the country code from your profile settings.<br>
                 * <strong>Example for Philippines</strong><br>
                 * E.164: +639184661533<br>
                 * Local: 09184661533
                 * @apiParam {String} name Name of contact
                 * @apiParam {String} groups List of contact group ID's separated by commas. You can get group ID's from <strong>/get/groups</strong> (Your contact groups).
                 *
                 * @apiExample {php} PHP Example
                  <?php

                    $contact = [
                        "secret" => "API_SECRET", // your API secret from (Tools -> API Keys) page
                        "groups" => "1,2,3,4",
                        "phone" => "+639123456789",
                        "name" => "Martin Crater"
                    ];

                    $cURL = curl_init("http://127.0.0.1/zender/api/create/contact");
                    curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    curl_setopt($cURL, CURLOPT_POSTFIELDS, $contact);
                    $response = curl_exec($cURL);
                    curl_close($cURL);

                    $result = json_decode($response, true);

                    // do something with response
                    print_r($result);
                 * 
                 * @apiExample {python} Python Example
                    import requests
    
                    # your API secret from (Tools -> API Keys) page
                    apiSecret = "API_SECRET"

                    contact = {
                        "secret": apiSecret,
                        "groups": "1,2,3,4",
                        "phone": "+639123456789",
                        "name": "Martin Crater"
                    }

                    r = requests.post(url = "http://127.0.0.1/zender/api/create/contact", params = contact)
                      
                    # do something with response object
                    result = r.json()
                 *
                 * @apiSuccess (Success Response Format) {Number} status List of Codes
                 * <br> 200 = Success
                 * @apiSuccess (Success Response Format) {String} message Response message
                 * @apiSuccess (Success Response Format) {Array} data Array of data
                 *
                 * @apiSuccessExample {json} Success Response
                 {
                   "status": 200,
                   "message": "Contact has been created!",
                   "data": false
                 }
                 * 
                 * @apiError (Error Response Format) {Number} status List of Codes<br>
                 * 400 = Invalid parameters<br>
                 * 401 = Invalid API secret<br>
                 * 403 = Access denied<br>
                 * 500 = Something went wrong
                 * @apiError (Error Response Format) {String} message Response message
                 * @apiError (Error Response Format) {Array} data Array of data
                 * 
                 * @apiErrorExample {json} Error Response
                 {
                   "status": 400,
                   "message": "Invalid Parameters!",
                   "data": false
                 }
                 *
                 */

                if(!in_array("create_contact", explode(",", $api["permissions"])))
                    response(403, "This API key doesn't have permission to use this endpoint!");

                if(!isset($request["name"], $request["phone"], $request["groups"]))
                    response(400, "Invalid Parameters!");

                $subscription = set_subscription(
                    $this->system->checkSubscription($api["uid"]), 
                    $this->system->getSubscription(false, $api["uid"]), 
                    $this->system->getSubscription(false, false, true)
                );

                if(empty($subscription))
                    response(400, "Account is not subscribed to any premium package!");

                if(limitation($subscription["contact_limit"], $this->system->countContacts($api["uid"])))
                    response(400, "You have reached the maximum allowed contacts!");

                if(!$this->sanitize->length($request["name"]))
                    response(400, "Contact name is too short!");

                try {
                    $number = $this->phone->parse($request["phone"], $api["country"]);

                    $number->format(Brick\PhoneNumber\PhoneNumberFormat::INTERNATIONAL);

                    if (!$number->isValidNumber())
                        response(400, "Invalid phone number!");

                    if(!$number->getNumberType(Brick\PhoneNumber\PhoneNumberType::MOBILE))
                        response(400, "Invalid phone number!");

                    $request["phone"] = $number->format(Brick\PhoneNumber\PhoneNumberFormat::E164);
                } catch(Brick\PhoneNumber\PhoneNumberParseException $e) {
                    response(400, "Invalid phone number!");
                }

                $groups = explode(",", $request["groups"]);

                if(!is_array($groups))
                    response(400, "Invalid Parameters!");

                foreach($groups as $group):
                    if($this->system->checkGroup($api["uid"], $group) < 1)
                        response(400, "Group #{$group} is invalid!");
                endforeach;

                if($this->system->checkNumber($api["uid"], $request["phone"]) > 0)
                    response(400, "Phone number already exist in your contact book!");

                $filtered = [
                    "uid" => $api["uid"],
                    "groups" => implode(",", $groups),
                    "phone" => $request["phone"],
                    "name" => $request["name"]
                ];

                if($this->system->create("contacts", $filtered)):
                    $this->cache->container("contacts.{$api["hash"]}");
                    $this->cache->clear();

                    try {
                        $echoToken = $this->echo->token();

                        if($echoToken):
                            $this->echo->notify($api["hash"], [
                                "type" => "table"
                            ]);
                        endif;
                    } catch(Exception $e){
                        response(500, "System configuration error!");
                    }

                    response(200, "Contact has been created!");
                else:
                    response(500, "Something went wrong!");
                endif;

                break;
            case "group":
                /**
                 * @api {post} /create/group Create Group
                 * @apiName Create Group
                 * @apiDescription Create Group. Requires "<strong>create_group</strong>" API permission.
                 * @apiGroup Contacts
                 * @apiVersion 1.0.0
                 *
                 * @apiParam {String} secret The API secret you copied from (Tools -> API Keys) page
                 * @apiParam {String} name Name of group
                 *
                 * @apiExample {php} PHP Example
                  <?php

                    $group = [
                        "secret" => "API_SECRET", // your API secret from (Tools -> API Keys) page
                        "name" => "Friends"
                    ];

                    $cURL = curl_init("http://127.0.0.1/zender/api/create/group");
                    curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    curl_setopt($cURL, CURLOPT_POSTFIELDS, $group);
                    $response = curl_exec($cURL);
                    curl_close($cURL);

                    $result = json_decode($response, true);

                    // do something with response
                    print_r($result);
                 * 
                 * @apiExample {python} Python Example
                    import requests
    
                    # your API secret from (Tools -> API Keys) page
                    apiSecret = "API_SECRET"

                    group = {
                        "secret": apiSecret,
                        "name": "Friends"
                    }

                    r = requests.post(url = "http://127.0.0.1/zender/api/create/group", params = group)
                      
                    # do something with response object
                    result = r.json()
                 *
                 * @apiSuccess (Success Response Format) {Number} status List of Codes
                 * <br> 200 = Success
                 * @apiSuccess (Success Response Format) {String} message Response message
                 * @apiSuccess (Success Response Format) {Array} data Array of data
                 *
                 * @apiSuccessExample {json} Success Response
                 {
                   "status": 200,
                   "message": "Contact group has been created!",
                   "data": false
                 }
                 * 
                 * @apiError (Error Response Format) {Number} status List of Codes<br>
                 * 400 = Invalid parameters<br>
                 * 401 = Invalid API secret<br>
                 * 403 = Access denied<br>
                 * 500 = Something went wrong
                 * @apiError (Error Response Format) {String} message Response message
                 * @apiError (Error Response Format) {Array} data Array of data
                 * 
                 * @apiErrorExample {json} Error Response
                 {
                   "status": 400,
                   "message": "Invalid Parameters!",
                   "data": false
                 }
                 *
                 */

                if(!in_array("create_group", explode(",", $api["permissions"])))
                    response(403, "This API key doesn't have permission to use this endpoint!");

                if(!isset($request["name"]))
                    response(400, "Invalid Parameters!");

                if(!$this->sanitize->length($request["name"]))
                    response(400, "Group name is too short!");

                $filtered = [
                    "uid" => $api["uid"],
                    "name" => $request["name"]
                ];

                if($this->system->create("groups", $filtered)):
                    try {
                        $echoToken = $this->echo->token();

                        if($echoToken):
                            $this->echo->notify($api["hash"], [
                                "type" => "table"
                            ]);
                        endif;
                    } catch(Exception $e){
                        response(500, "System configuration error!");
                    }

                    response(200, "Contact group has been created!");
                else:
                    response(500, "Something went wrong!");
                endif;

                break;
            default:
                response(400, "Invalid API Endpoint!");
        endswitch;
	}

	public function delete()
	{
		$this->header->allow();

        $this->cache->container("system.settings");

        if($this->cache->empty()):
            $this->cache->setArray($this->system->getSettings());
        endif;

        set_system($this->cache->getAll());

        $request = $this->sanitize->array($_REQUEST);
        $type = $this->sanitize->string($this->url->segment(4));

        if(!isset($request["secret"]))
            response(400, "Invalid Parameters!");

        if($this->api->checkApikey($request["secret"]) < 1)
            response(401, "Invalid API secret supplied!");

        $api = $this->api->getApikey($request["secret"]);

        $permissions = explode(",", $api["permissions"]);

        if(!is_array($permissions))
            response(403, "Insufficient Permissions!");

        $this->echo->_cache = $this->cache;
        $this->echo->_guzzle = $this->guzzle;

        switch($type):
            case "contact":
                /**
                 * @api {get} /delete/contact Delete Contact
                 * @apiName Delete Contact
                 * @apiDescription Delete Contact. Requires "<strong>delete_contact</strong>" API permission.
                 * @apiGroup Contacts
                 * @apiVersion 1.0.0
                 *
                 * @apiParam {String} secret The API secret you copied from (Tools -> API Keys) page
                 * @apiParam {Number} id Contact ID
                 *
                 * @apiExample {php} PHP Example
                  <?php

                    $apiSecret = "API_SECRET"; // your API secret from (Tools -> API Keys) page
                    $contactId = 1;

                    $cURL = curl_init();
                    curl_setopt($cURL, CURLOPT_URL, "http://127.0.0.1/zender/api/delete/contact?secret={$apiSecret}&id={$contactId}");
                    curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    $response = curl_exec($cURL);
                    curl_close($cURL);

                    $result = json_decode($response, true);

                    // do something with response
                    print_r($result);
                 * 
                 * @apiExample {python} Python Example
                    import requests
    
                    # your API secret from (Tools -> API Keys) page
                    apiSecret = "API_SECRET"
                    contactId = 1

                    r = requests.get(url = "http://127.0.0.1/zender/api/delete/contact", params = {
                        "secret": apiSecret
                        "id": contactId
                    })
                      
                    # do something with response object
                    result = r.json()
                 * 
                 * @apiSuccess (Success Response Format) {Number} status List of Codes
                 * <br> 200 = Success
                 * @apiSuccess (Success Response Format) {String} message Response message
                 * @apiSuccess (Success Response Format) {Array} data Array of data
                 *
                 * @apiSuccessExample {json} Success Response
                 {
                   "status": 200,
                   "message": "Contact has been deleted!",
                   "data": false
                 }
                 * 
                 * @apiError (Error Response Format) {Number} status List of Codes<br>
                 * 400 = Invalid parameters<br>
                 * 401 = Invalid API secret<br>
                 * 403 = Access denied<br>
                 * 500 = Something went wrong
                 * @apiError (Error Response Format) {String} message Response message
                 * @apiError (Error Response Format) {Array} data Array of data
                 * 
                 * @apiErrorExample {json} Error Response
                 {
                   "status": 400,
                   "message": "Invalid Parameters!",
                   "data": false
                 }
                 *
                 */

                if(!in_array("delete_contact", explode(",", $api["permissions"])))
                    response(403, "This API key doesn't have permission to use this endpoint!");

                if(!isset($request["id"]))
                    response(400, "Invalid Parameters!");

                if(!$this->sanitize->isInt($request["id"]))
                    response(400, "Invalid Parameters!");

                if($this->system->delete($api["uid"], $request["id"], "contacts")):
                    try {
                        $echoToken = $this->echo->token();

                        if($echoToken):
                            $this->echo->notify($api["hash"], [
                                "type" => "table"
                            ]);
                        endif;
                    } catch(Exception $e){
                        response(500, "System configuration error!");
                    }

                    response(200, "Contact has been deleted!");
                else:
                    response(500, "Something went wrong!");
                endif;

                break;
            case "group":
                /**
                 * @api {get} /delete/group Delete Group
                 * @apiName Delete Group
                 * @apiDescription Delete Group. Requires "<strong>delete_group</strong>" API permission.
                 * @apiGroup Contacts
                 * @apiVersion 1.0.0
                 *
                 * @apiParam {String} secret The API secret you copied from (Tools -> API Keys) page
                 * @apiParam {Number} id Contact group ID
                 *
                 * @apiExample {php} PHP Example
                  <?php

                    $apiSecret = "API_SECRET"; // your API secret from (Tools -> API Keys) page
                    $groupId = 1;

                    $cURL = curl_init();
                    curl_setopt($cURL, CURLOPT_URL, "http://127.0.0.1/zender/api/delete/group?secret={$apiSecret}&id={$groupId}");
                    curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    $response = curl_exec($cURL);
                    curl_close($cURL);

                    $result = json_decode($response, true);

                    // do something with response
                    print_r($result);
                 * 
                 * @apiExample {python} Python Example
                    import requests
    
                    # your API secret from (Tools -> API Keys) page
                    apiSecret = "API_SECRET"
                    groupId = 1

                    r = requests.get(url = "http://127.0.0.1/zender/api/delete/group", params = {
                        "secret": apiSecret
                        "id": groupId
                    })
                      
                    # do something with response object
                    result = r.json()
                 *
                 * @apiSuccess (Success Response Format) {Number} status List of Codes
                 * <br> 200 = Success
                 * @apiSuccess (Success Response Format) {String} message Response message
                 * @apiSuccess (Success Response Format) {Array} data Array of data
                 *
                 * @apiSuccessExample {json} Success Response
                 {
                   "status": 200,
                   "message": "Contact group has been deleted!",
                   "data": false
                 }
                 * 
                 * @apiError (Error Response Format) {Number} status List of Codes<br>
                 * 400 = Invalid parameters<br>
                 * 401 = Invalid API secret<br>
                 * 403 = Access denied<br>
                 * 500 = Something went wrong
                 * @apiError (Error Response Format) {String} message Response message
                 * @apiError (Error Response Format) {Array} data Array of data
                 * 
                 * @apiErrorExample {json} Error Response
                 {
                   "status": 400,
                   "message": "Invalid Parameters!",
                   "data": false
                 }
                 *
                 */

                if(!in_array("delete_group", explode(",", $api["permissions"])))
                    response(403, "This API key doesn't have permission to use this endpoint!");

                if(!isset($request["id"]))
                    response(400, "Invalid Parameters!");

                if(!$this->sanitize->isInt($request["id"]))
                    response(400, "Invalid Parameters!");

                if($this->system->delete($api["uid"], $request["id"], "groups")):
                    try {
                        $echoToken = $this->echo->token();

                        if($echoToken):
                            $this->echo->notify($api["hash"], [
                                "type" => "table"
                            ]);
                        endif;
                    } catch(Exception $e){
                        response(500, "System configuration error!");
                    }

                    response(200, "Contact group has been deleted!");
                else:
                    response(500, "Something went wrong!");
                endif;

                break;
            case "unsubscribed":
                /**
                 * @api {get} /delete/unsubscribed Delete Unsubscribed
                 * @apiName Delete Unsubscribed
                 * @apiDescription Delete Unsubscribed Contact. Requires "<strong>delete_unsubscribed</strong>" API permission.
                 * @apiGroup Contacts
                 * @apiVersion 1.0.0
                 *
                 * @apiParam {String} secret The API secret you copied from (Tools -> API Keys) page
                 * @apiParam {Number} id Contact group ID
                 *
                 * @apiExample {php} PHP Example
                  <?php

                    $apiSecret = "API_SECRET"; // your API secret from (Tools -> API Keys) page
                    $unsubscribedId = 1;

                    $cURL = curl_init();
                    curl_setopt($cURL, CURLOPT_URL, "http://127.0.0.1/zender/api/delete/unsubscribed?secret={$apiSecret}&id={$unsubscribedId}");
                    curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    $response = curl_exec($cURL);
                    curl_close($cURL);

                    $result = json_decode($response, true);

                    // do something with response
                    print_r($result);
                 * 
                 * @apiExample {python} Python Example
                    import requests
    
                    # your API secret from (Tools -> API Keys) page
                    apiSecret = "API_SECRET"
                    unsubscribedId = 1

                    r = requests.get(url = "http://127.0.0.1/zender/api/delete/unsubscribed", params = {
                        "secret": apiSecret
                        "id": unsubscribedId
                    })
                      
                    # do something with response object
                    result = r.json()
                 *
                 * @apiSuccess (Success Response Format) {Number} status List of Codes
                 * <br> 200 = Success
                 * @apiSuccess (Success Response Format) {String} message Response message
                 * @apiSuccess (Success Response Format) {Array} data Array of data
                 *
                 * @apiSuccessExample {json} Success Response
                 {
                   "status": 200,
                   "message": "Unsubscribed contact has been deleted!",
                   "data": false
                 }
                 * 
                 * @apiError (Error Response Format) {Number} status List of Codes<br>
                 * 400 = Invalid parameters<br>
                 * 401 = Invalid API secret<br>
                 * 403 = Access denied<br>
                 * 500 = Something went wrong
                 * @apiError (Error Response Format) {String} message Response message
                 * @apiError (Error Response Format) {Array} data Array of data
                 * 
                 * @apiErrorExample {json} Error Response
                 {
                   "status": 400,
                   "message": "Invalid Parameters!",
                   "data": false
                 }
                 *
                 */

                if(!in_array("delete_unsubscribed", explode(",", $api["permissions"])))
                    response(403, "This API key doesn't have permission to use this endpoint!");

                if(!isset($request["id"]))
                    response(400, "Invalid Parameters!");

                if(!$this->sanitize->isInt($request["id"]))
                    response(400, "Invalid Parameters!");

                if($this->system->delete($api["uid"], $request["id"], "unsubscribed")):
                    try {
                        $echoToken = $this->echo->token();

                        if($echoToken):
                            $this->echo->notify($api["hash"], [
                                "type" => "table"
                            ]);
                        endif;
                    } catch(Exception $e){
                        response(500, "System configuration error!");
                    }

                    response(200, "Unsubscribed contact has been deleted!");
                else:
                    response(500, "Something went wrong!");
                endif;

                break;
            case "sms.sent":
                /**
                 * @api {get} /delete/sms.sent Delete Sent Message
                 * @apiName Delete Sent Message
                 * @apiDescription Delete Sent Message. Requires "<strong>delete_sms_sent</strong>" API permission.
                 * @apiGroup SMS
                 * @apiVersion 1.0.0
                 *
                 * @apiParam {String} secret The API secret you copied from (Tools -> API Keys) page
                 * @apiParam {Number} id Sent message ID
                 *
                 * @apiExample {php} PHP Example
                  <?php

                    $apiSecret = "API_SECRET"; // your API secret from (Tools -> API Keys) page
                    $smsId = 1;

                    $cURL = curl_init();
                    curl_setopt($cURL, CURLOPT_URL, "http://127.0.0.1/zender/api/delete/sms.sent?secret={$apiSecret}&id={$smsId}");
                    curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    $response = curl_exec($cURL);
                    curl_close($cURL);

                    $result = json_decode($response, true);

                    // do something with response
                    print_r($result);
                 * 
                 * @apiExample {python} Python Example
                    import requests
    
                    # your API secret from (Tools -> API Keys) page
                    apiSecret = "API_SECRET"
                    smsId = 1

                    r = requests.get(url = "http://127.0.0.1/zender/api/delete/sms.sent", params = {
                        "secret": apiSecret
                        "id": smsId
                    })
                      
                    # do something with response object
                    result = r.json()
                 *
                 * @apiSuccess (Success Response Format) {Number} status List of Codes
                 * <br> 200 = Success
                 * @apiSuccess (Success Response Format) {String} message Response message
                 * @apiSuccess (Success Response Format) {Array} data Array of data
                 *
                 * @apiSuccessExample {json} Success Response
                 {
                   "status": 200,
                   "message": "Sent SMS has been deleted!",
                   "data": false
                 }
                 * 
                 * @apiError (Error Response Format) {Number} status List of Codes<br>
                 * 400 = Invalid parameters<br>
                 * 401 = Invalid API secret<br>
                 * 403 = Access denied<br>
                 * 500 = Something went wrong
                 * @apiError (Error Response Format) {String} message Response message
                 * @apiError (Error Response Format) {Array} data Array of data
                 * 
                 * @apiErrorExample {json} Error Response
                 {
                   "status": 400,
                   "message": "Invalid Parameters!",
                   "data": false
                 }
                 *
                 */

                if(!in_array("delete_sms_sent", explode(",", $api["permissions"])))
                    response(403, "This API key doesn't have permission to use this endpoint!");

                if(!isset($request["id"]))
                    response(400, "Invalid Parameters!");

                if(!$this->sanitize->isInt($request["id"]))
                    response(400, "Invalid Parameters!");

                if($this->system->delete($api["uid"], $request["id"], "sent")):
                    try {
                        $echoToken = $this->echo->token();

                        if($echoToken):
                            $this->echo->notify($api["hash"], [
                                "type" => "table"
                            ]);
                        endif;
                    } catch(Exception $e){
                        response(500, "System configuration error!");
                    }

                    response(200, "Sent SMS has been deleted!");
                else:
                    response(500, "Something went wrong!");
                endif;

                break;
            case "sms.received":
                /**
                 * @api {get} /delete/sms.received Delete Received Message
                 * @apiName Delete Received Message
                 * @apiDescription Delete Received Message. Requires "<strong>delete_sms_received</strong>" API permission.
                 * @apiGroup SMS
                 * @apiVersion 1.0.0
                 *
                 * @apiParam {String} secret The API secret you copied from (Tools -> API Keys) page
                 * @apiParam {Number} id Received message ID
                 *
                 * @apiExample {php} PHP Example
                  <?php

                    $apiSecret = "API_SECRET"; // your API secret from (Tools -> API Keys) page
                    $smsId = 1;

                    $cURL = curl_init();
                    curl_setopt($cURL, CURLOPT_URL, "http://127.0.0.1/zender/api/delete/sms.received?secret={$apiSecret}&id={$smsId}");
                    curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    $response = curl_exec($cURL);
                    curl_close($cURL);

                    $result = json_decode($response, true);

                    // do something with response
                    print_r($result);
                 * 
                 * @apiExample {python} Python Example
                    import requests
    
                    # your API secret from (Tools -> API Keys) page
                    apiSecret = "API_SECRET"
                    smsId = 1

                    r = requests.get(url = "http://127.0.0.1/zender/api/delete/sms.received", params = {
                        "secret": apiSecret
                        "id": smsId
                    })
                      
                    # do something with response object
                    result = r.json()
                 *
                 * @apiSuccess (Success Response Format) {Number} status List of Codes
                 * <br> 200 = Success
                 * @apiSuccess (Success Response Format) {String} message Response message
                 * @apiSuccess (Success Response Format) {Array} data Array of data
                 *
                 * @apiSuccessExample {json} Success Response
                 {
                   "status": 200,
                   "message": "Received SMS has been deleted!",
                   "data": false
                 }
                 * 
                 * @apiError (Error Response Format) {Number} status List of Codes<br>
                 * 400 = Invalid parameters<br>
                 * 401 = Invalid API secret<br>
                 * 403 = Access denied<br>
                 * 500 = Something went wrong
                 * @apiError (Error Response Format) {String} message Response message
                 * @apiError (Error Response Format) {Array} data Array of data
                 * 
                 * @apiErrorExample {json} Error Response
                 {
                   "status": 400,
                   "message": "Invalid Parameters!",
                   "data": false
                 }
                 *
                 */

                if(!in_array("delete_sms_received", explode(",", $api["permissions"])))
                    response(403, "This API key doesn't have permission to use this endpoint!");

                if(!isset($request["id"]))
                    response(400, "Invalid Parameters!");

                if(!$this->sanitize->isInt($request["id"]))
                    response(400, "Invalid Parameters!");

                if($this->system->delete($api["uid"], $request["id"], "received")):
                    try {
                        $echoToken = $this->echo->token();

                        if($echoToken):
                            $this->echo->notify($api["hash"], [
                                "type" => "table"
                            ]);
                        endif;
                    } catch(Exception $e){
                        response(500, "System configuration error!");
                    }

                    response(200, "Received SMS has been deleted!");
                else:
                    response(500, "Something went wrong!");
                endif;

                break;
            case "wa.sent":
                /**
                 * @api {get} /delete/wa.sent Delete Sent Chat
                 * @apiName Delete Sent Chat
                 * @apiDescription Delete Sent Chat. Requires "<strong>delete_wa_sent</strong>" API permission.
                 * @apiGroup WhatsApp
                 * @apiVersion 1.0.0
                 *
                 * @apiParam {String} secret The API secret you copied from (Tools -> API Keys) page
                 * @apiParam {Number} id Sent chat ID
                 *
                 * @apiExample {php} PHP Example
                  <?php

                    $apiSecret = "API_SECRET"; // your API secret from (Tools -> API Keys) page
                    $chatId = 1;

                    $cURL = curl_init();
                    curl_setopt($cURL, CURLOPT_URL, "http://127.0.0.1/zender/api/delete/wa.sent?secret={$apiSecret}&id={$chatId}");
                    curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    $response = curl_exec($cURL);
                    curl_close($cURL);

                    $result = json_decode($response, true);

                    // do something with response
                    print_r($result);
                 * 
                 * @apiExample {python} Python Example
                    import requests
    
                    # your API secret from (Tools -> API Keys) page
                    apiSecret = "API_SECRET"
                    chatId = 1

                    r = requests.get(url = "http://127.0.0.1/zender/api/delete/wa.sent", params = {
                        "secret": apiSecret
                        "id": chatId
                    })
                      
                    # do something with response object
                    result = r.json()
                 *
                 * @apiSuccess (Success Response Format) {Number} status List of Codes
                 * <br> 200 = Success
                 * @apiSuccess (Success Response Format) {String} message Response message
                 * @apiSuccess (Success Response Format) {Array} data Array of data
                 *
                 * @apiSuccessExample {json} Success Response
                 {
                   "status": 200,
                   "message": "Sent WhatsApp chat has been deleted!",
                   "data": false
                 }
                 * 
                 * @apiError (Error Response Format) {Number} status List of Codes<br>
                 * 400 = Invalid parameters<br>
                 * 401 = Invalid API secret<br>
                 * 403 = Access denied<br>
                 * 500 = Something went wrong
                 * @apiError (Error Response Format) {String} message Response message
                 * @apiError (Error Response Format) {Array} data Array of data
                 * 
                 * @apiErrorExample {json} Error Response
                 {
                   "status": 400,
                   "message": "Invalid Parameters!",
                   "data": false
                 }
                 *
                 */

                if(!in_array("delete_wa_sent", explode(",", $api["permissions"])))
                    response(403, "This API key doesn't have permission to use this endpoint!");

                if(!isset($request["id"]))
                    response(400, "Invalid Parameters!");

                if(!$this->sanitize->isInt($request["id"]))
                    response(400, "Invalid Parameters!");

                if($this->system->delete($api["uid"], $request["id"], "wa_sent")):
                    try {
                        $echoToken = $this->echo->token();

                        if($echoToken):
                            $this->echo->notify($api["hash"], [
                                "type" => "table"
                            ]);
                        endif;
                    } catch(Exception $e){
                        response(500, "System configuration error!");
                    }

                    response(200, "Sent WhatsApp chat has been deleted!");
                else:
                    response(500, "Something went wrong!");
                endif;

                break;
            case "wa.received":
                /**
                 * @api {get} /delete/wa.received Delete Received Chat
                 * @apiName Delete Received Chat
                 * @apiDescription Delete Received Chat. Requires "<strong>delete_wa_received</strong>" API permission.
                 * @apiGroup WhatsApp
                 * @apiVersion 1.0.0
                 *
                 * @apiParam {String} secret The API secret you copied from (Tools -> API Keys) page
                 * @apiParam {Number} id Received chat ID
                 *
                 * @apiExample {php} PHP Example
                  <?php

                    $apiSecret = "API_SECRET"; // your API secret from (Tools -> API Keys) page
                    $chatId = 1;

                    $cURL = curl_init();
                    curl_setopt($cURL, CURLOPT_URL, "http://127.0.0.1/zender/api/delete/wa.received?secret={$apiSecret}&id={$chatId}");
                    curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    $response = curl_exec($cURL);
                    curl_close($cURL);

                    $result = json_decode($response, true);

                    // do something with response
                    print_r($result);
                 * 
                 * @apiExample {python} Python Example
                    import requests
    
                    # your API secret from (Tools -> API Keys) page
                    apiSecret = "API_SECRET"
                    chatId = 1

                    r = requests.get(url = "http://127.0.0.1/zender/api/delete/wa.received", params = {
                        "secret": apiSecret
                        "id": chatId
                    })
                      
                    # do something with response object
                    result = r.json()
                 *
                 * @apiSuccess (Success Response Format) {Number} status List of Codes
                 * <br> 200 = Success
                 * @apiSuccess (Success Response Format) {String} message Response message
                 * @apiSuccess (Success Response Format) {Array} data Array of data
                 *
                 * @apiSuccessExample {json} Success Response
                 {
                   "status": 200,
                   "message": "Received WhatsApp chat has been deleted!",
                   "data": false
                 }
                 * 
                 * @apiError (Error Response Format) {Number} status List of Codes<br>
                 * 400 = Invalid parameters<br>
                 * 401 = Invalid API secret<br>
                 * 403 = Access denied<br>
                 * 500 = Something went wrong
                 * @apiError (Error Response Format) {String} message Response message
                 * @apiError (Error Response Format) {Array} data Array of data
                 * 
                 * @apiErrorExample {json} Error Response
                 {
                   "status": 400,
                   "message": "Invalid Parameters!",
                   "data": false
                 }
                 *
                 */

                if(!in_array("delete_wa_received", explode(",", $api["permissions"])))
                    response(403, "This API key doesn't have permission to use this endpoint!");

                if(!isset($request["id"]))
                    response(400, "Invalid Parameters!");

                if(!$this->sanitize->isInt($request["id"]))
                    response(400, "Invalid Parameters!");

                if($this->system->delete($api["uid"], $request["id"], "wa_received")):
                    try {
                        $echoToken = $this->echo->token();

                        if($echoToken):
                            $this->echo->notify($api["hash"], [
                                "type" => "table"
                            ]);
                        endif;
                    } catch(Exception $e){
                        response(500, "System configuration error!");
                    }

                    response(200, "Received WhatsApp chat has been deleted!");
                else:
                    response(500, "Something went wrong!");
                endif;

                break;
            case "ussd":
                /**
                 * @api {get} /delete/ussd Delete USSD Request
                 * @apiName Delete USSD Request
                 * @apiDescription Delete USSD Request. Requires "<strong>delete_ussd</strong>" API permission.
                 * @apiGroup USSD
                 * @apiVersion 1.0.0
                 *
                 * @apiParam {String} secret The API secret you copied from (Tools -> API Keys) page
                 * @apiParam {Number} id USSD request ID
                 *
                 * @apiExample {php} PHP Example
                  <?php

                    $apiSecret = "API_SECRET"; // your API secret from (Tools -> API Keys) page
                    $ussdId = 1;

                    $cURL = curl_init();
                    curl_setopt($cURL, CURLOPT_URL, "http://127.0.0.1/zender/api/delete/ussd?secret={$apiSecret}&id={$ussdId}");
                    curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    $response = curl_exec($cURL);
                    curl_close($cURL);

                    $result = json_decode($response, true);

                    // do something with response
                    print_r($result);
                 * 
                 * @apiExample {python} Python Example
                    import requests
    
                    # your API secret from (Tools -> API Keys) page
                    apiSecret = "API_SECRET"
                    ussdId = 1

                    r = requests.get(url = "http://127.0.0.1/zender/api/delete/ussd", params = {
                        "secret": apiSecret
                        "id": ussdId
                    })
                      
                    # do something with response object
                    result = r.json()
                 *
                 * @apiSuccess (Success Response Format) {Number} status List of Codes
                 * <br> 200 = Success
                 * @apiSuccess (Success Response Format) {String} message Response message
                 * @apiSuccess (Success Response Format) {Array} data Array of data
                 *
                 * @apiSuccessExample {json} Success Response
                 {
                   "status": 200,
                   "message": "USSD request has been deleted!",
                   "data": false
                 }
                 * 
                 * @apiError (Error Response Format) {Number} status List of Codes<br>
                 * 400 = Invalid parameters<br>
                 * 401 = Invalid API secret<br>
                 * 403 = Access denied<br>
                 * 500 = Something went wrong
                 * @apiError (Error Response Format) {String} message Response message
                 * @apiError (Error Response Format) {Array} data Array of data
                 * 
                 * @apiErrorExample {json} Error Response
                 {
                   "status": 400,
                   "message": "Invalid Parameters!",
                   "data": false
                 }
                 *
                 */

                if(!in_array("delete_ussd", explode(",", $api["permissions"])))
                    response(403, "This API key doesn't have permission to use this endpoint!");

                if(!isset($request["id"]))
                    response(400, "Invalid Parameters!");

                if(!$this->sanitize->isInt($request["id"]))
                    response(400, "Invalid Parameters!");

                if($this->system->delete($api["uid"], $request["id"], "ussd")):
                    try {
                        $echoToken = $this->echo->token();

                        if($echoToken):
                            $this->echo->notify($api["hash"], [
                                "type" => "table"
                            ]);
                        endif;
                    } catch(Exception $e){
                        response(500, "System configuration error!");
                    }

                    response(200, "USSD eequest has been deleted!");
                else:
                    response(500, "Something went wrong!");
                endif;

                break;
            case "notification":
                /**
                 * @api {get} /delete/notification Delete Notification
                 * @apiName Delete Notification
                 * @apiDescription Delete Notification. Requires "<strong>delete_notification</strong>" API permission.
                 * @apiGroup Notifications
                 * @apiVersion 1.0.0
                 *
                 * @apiParam {String} secret The API secret you copied from (Tools -> API Keys) page
                 * @apiParam {Number} id Notification ID
                 *
                 * @apiExample {php} PHP Example
                  <?php

                    $apiSecret = "API_SECRET"; // your API secret from (Tools -> API Keys) page
                    $notificationId = 1;

                    $cURL = curl_init();
                    curl_setopt($cURL, CURLOPT_URL, "http://127.0.0.1/zender/api/delete/notification?secret={$apiSecret}&id={$notificationId}");
                    curl_setopt($cURL, CURLOPT_RETURNTRANSFER, true);
                    $response = curl_exec($cURL);
                    curl_close($cURL);

                    $result = json_decode($response, true);

                    // do something with response
                    print_r($result);
                 * 
                 * @apiExample {python} Python Example
                    import requests
    
                    # your API secret from (Tools -> API Keys) page
                    apiSecret = "API_SECRET"
                    notificationId = 1

                    r = requests.get(url = "http://127.0.0.1/zender/api/delete/notification", params = {
                        "secret": apiSecret
                        "id": notificationId
                    })
                      
                    # do something with response object
                    result = r.json()
                 *
                 * @apiSuccess (Success Response Format) {Number} status List of Codes
                 * <br> 200 = Success
                 * @apiSuccess (Success Response Format) {String} message Response message
                 * @apiSuccess (Success Response Format) {Array} data Array of data
                 *
                 * @apiSuccessExample {json} Success Response
                 {
                   "status": 200,
                   "message": "Notification has been deleted!",
                   "data": false
                 }
                 * 
                 * @apiError (Error Response Format) {Number} status List of Codes<br>
                 * 400 = Invalid parameters<br>
                 * 401 = Invalid API secret<br>
                 * 403 = Access denied<br>
                 * 500 = Something went wrong
                 * @apiError (Error Response Format) {String} message Response message
                 * @apiError (Error Response Format) {Array} data Array of data
                 * 
                 * @apiErrorExample {json} Error Response
                 {
                   "status": 400,
                   "message": "Invalid Parameters!",
                   "data": false
                 }
                 *
                 */

                if(!in_array("delete_notification", explode(",", $api["permissions"])))
                    response(403, "This API key doesn't have permission to use this endpoint!");

                if(!isset($request["id"]))
                    response(400, "Invalid Parameters!");

                if(!$this->sanitize->isInt($request["id"]))
                    response(400, "Invalid Parameters!");

                if($this->system->delete($api["uid"], $request["id"], "notifications")):
                    try {
                        $echoToken = $this->echo->token();

                        if($echoToken):
                            $this->echo->notify($api["hash"], [
                                "type" => "table"
                            ]);
                        endif;
                    } catch(Exception $e){
                        response(500, "System configuration error!");
                    }

                    response(200, "Notification has been deleted!");
                else:
                    response(500, "Something went wrong!");
                endif;

                break;
            default:
                response(400, "Invalid API Endpoint!");
        endswitch;
	}
}